<?php
  /**
   * @package     Joomla.Administrator
   * @subpackage  com_formea
   *
   * @copyright   Copyright (C) 2010-2024 Feseur Sdn Bhd. All rights reserved.
   * @license     GNU General Public License version 2 or later; see LICENSE.txt
   * @version     1.2.2
	 */

  namespace Joomla\Component\Formea\Administrator\Helper;

  defined('_JEXEC') or die;

  use Joomla\CMS\Factory;
  use Joomla\Filesystem\Path;
  use Joomla\CMS\Helper\ContentHelper;
  use Joomla\CMS\Language\Text;
  use Joomla\CMS\Router\Route;
  use Joomla\CMS\Uri\Uri;
  use Joomla\Component\Formea\Site\Helper\FormeaGeneralHelper;
  use Joomla\Component\Formea\Site\Libraries\FormeaElement;
  use Joomla\Component\Formea\Site\Libraries\FormeaGroupedElement;
  use Joomla\Database\DatabaseDriver;
  use Joomla\Registry\Registry;
  use stdClass;

  /**
   * Formea component helper.
   *
   * @since  1.0.0
   */
  class FormeaHelper extends ContentHelper
  {

    /**
     * Method to parse XML for Element Detail
     *
     * @param $lang_code
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function commonElementDetailXml($lang_code)
    {
      $lang_code = str_replace('-', '_', $lang_code);
      $con       = '    <fields name="' . $lang_code . '">' . "\r\n";
      $con       .= '      <field name="caption"  label="COM_FORMEA_CAPTION_LBL" description="COM_FORMEA_CAPTION_LBL_DESC" type="text"/>' . "\r\n";
      $con       .= '      <field name="placeholder"  label="COM_FORMEA_PLACEHOLDER" type="text"/>' . "\r\n";
      //$con       .= '      <field name="description"  label="COM_FORMEA_DESCRIPTION" type="editor" buttons="true"  hide="readmore,pagebreak"/>' . "\r\n";
      $con .= '      <field name="description"  label="COM_FORMEA_DESCRIPTION" type="textarea" />' . "\r\n";
      $con .= '    </fields>' . "\r\n";

      return $con;
    }

    /**
     * Method to standardized the Element Detail
     *
     * @param $lang_code
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function commonElementDetailArray($lang_code)
    {
      $lang_code        = str_replace('-', '_', $lang_code);
      $ret [$lang_code] = [
        'caption'     => '',
        'placeholder' => '',
        'description' => ''
      ];

      return $ret;
    }

    /**
     * General XML for element values
     *
     * @param   string       $lang_code
     * @param   null|object  $element_type
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function commonElementValueXml($lang_code, $element_config = null)
    {

      $valueType = 0;
      if (isset($element_config->values->valueType))
      {
        $valueType = (int) $element_config->values->valueType;
      }
      $limit = 0;
      if (isset($element_config->values->limit))
      {
        $limit = (int) $element_config->values->limit;
      }

      $lang_code = str_replace('-', '_', $lang_code);

      $multiType = [1, 2];

      if (in_array((int) $valueType, $multiType))
      {
        $isMulti = $valueType == 2;
        $field   = '<field name="option_lists" type="FormeaOptionVal" ';
        $field   .= 'multiple="' . $isMulti . '" ';
        $field   .= 'limit="' . $limit . '" ';
        $field   .= 'labelvalue="COM_FORMEA_VALUE" ';
        $field   .= 'labeltext="COM_FORMEA_TEXT" ';
        $field   .= 'labelselected="COM_FORMEA_SELECTED" ';
        $field   .= 'labelno="JNO" ';
        $field   .= 'labelyes="JYES" ';
        $field   .= '/>';
        $field   .= '<field name="defval"  type="hidden"/>';
      }
      elseif ($valueType == 3)
      {
        //textarea
        $field = '';
      }
      elseif ($valueType == 4)
      {
        //editor
        $field = '<field name="defval"  label="COM_FORMEA_DEFAULT_VALUE" description="COM_FORMEA_DEFAULT_VALUE_DESC" type="editor" buttons="true" filter="RAW"/>';
        // $field .= '<field name="option_lists"  type="hidden"/>';
      }
      elseif ($valueType == 7)
      {
        //multiple editors
        $field = '<field name="defval"  type="hidden"/>';
        $field .= '<field name="option_lists" type="subform" label="COM_FORMEA_OPTIONS" multiple="true" layout="joomla.form.field.subform.repeatable">';
        $field .= '<form>';
        $field .= '<field name="val" type="text" label="COM_FORMEA_VALUE"/>';
        $field .= '<field name="selected" type="radio" layout="joomla.form.field.radio.switcher" default="0"  label="COM_FORMEA_SELECTED">';
        $field .= '<option value="0">JNO</option>';
        $field .= '<option value="1">JYES</option>';
        $field .= '</field>';
        $field .= '<field name="text" type="editor" filter="safehtml" buttons="true" hide="pagebreak,readmore,fields" height="250"/>';
        $field .= '</form>';
        $field .= '</field>';
      }
      else
      {
        //default defval
        $field = '<field name="defval"  label="COM_FORMEA_DEFAULT_VALUE" description="COM_FORMEA_DEFAULT_VALUE_DESC" type="text"/>';
        // $field .= '<field name="option_lists"  type="hidden"/>';
      }

      $con = '    <fields name="' . $lang_code . '">' . "\r\n";
      $con .= '      ' . $field . "\r\n";
      $con .= '    </fields>' . "\r\n";

      return $con;
    }

    /**
     * @param $lang_code
     * @param $element_config
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function commonElementValueArray($lang_code, $element_config = null)
    {
      $lang_code        = str_replace('-', '_', $lang_code);
      $ret [$lang_code] = [
        'defval'       => '',
        'option_lists' => [],
      ];

      return $ret;
    }

    /**
     * Method to map the selected validations
     *
     * @param $validationLists
     * @param $elValidationsArray
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function mapSelectedValidations($validationLists, $elValidationsArray)
    {

      $selectedValidation = [];
      if (!empty($validationLists))
      {
        $totalLists    = count($validationLists);
        $elValidations = FormeaGeneralHelper::formatKeyArray($elValidationsArray, 'validation_type');
        for ($i = 0; $i < $totalLists; $i++)
        {
          if (isset($elValidations[$validationLists[$i]->name]))
          {
            $tgt = $elValidations[$validationLists[$i]->name];
            $t   = $validationLists[$i];
            if (isset($t->configs) && !empty($t->configs))
            {
              $totalC = count($t->configs);
              for ($j = 0; $j < $totalC; $j++)
              {
                $name = $t->configs[$j]->name;
                if (isset($tgt[$name]))
                {
                  $val = $tgt[$name];
                  if ($t->configs[$j]->translatable)
                  {
                    $totalLangVal = count($t->configs[$j]->language_val);
                    for ($l = 0; $l < $totalLangVal; $l++)
                    {
                      $lang_code = $t->configs[$j]->language_val[$l]->lang_code;

                      if (isset($val[$lang_code]))
                      {
                        $t->configs[$j]->language_val[$l]->value = $val[$lang_code];
                      }
                    }
                  }
                  else
                  {
                    $t->configs[$j]->value = $val;
                  }
                }
              }
            }
            $selectedValidation[] = $t;
          }
        }
      }

      return $selectedValidation;
    }

    /**
     * Method to get Formea Main Menus
     *
     * @param $section
     *
     * @return array|object[]
     *
     * @since 1.0.0
     */
    public static function getFormeaMenus($section = [])
    {
      $ret              = [];
      $manageItems      = [
        (object) [
          'icon'        => 'fab fa-wpforms',
          'title'       => Text::_('COM_FORMEA_MANAGE_FORMS'),
          'short_title' => Text::_('COM_FORMEA_FORMS'),
          'link'        => Route::_('index.php?option=com_formea&view=formeas'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'far fa-object-group',
          'title'       => Text::_('COM_FORMEA_GROUPED_ELEMENTS_MANAGE'),
          'short_title' => Text::_('COM_FORMEA_GROUPED_ELEMENTS'),
          'link'        => Route::_('index.php?option=com_formea&view=groupedelements'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'fas fa-stream',
          'title'       => Text::_('COM_FORMEA_MANAGE_ELEMENTS'),
          'short_title' => Text::_('COM_FORMEA_ELEMENTS'),
          'link'        => Route::_('index.php?option=com_formea&view=elements'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'far fa-hdd',
          'title'       => Text::_('COM_FORMEA_MANAGE_SUBMISSIONS'),
          'short_title' => Text::_('COM_FORMEA_SUBMISSIONS'),
          'link'        => Route::_('index.php?option=com_formea&view=submissions'),
          'internal'    => true,
        ],

      ];
      $installableItems = [
        (object) [
          'icon'        => 'fas fa-spell-check',
          'title'       => Text::_('COM_FORMEA_MANAGE_VALIDATIONS'),
          'short_title' => Text::_('COM_FORMEA_VALIDATIONS'),
          'link'        => Route::_('index.php?option=com_formea&view=validations'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'far fa-list-alt',
          'title'       => Text::_('COM_FORMEA_MANAGE_ELEMENT_TYPES'),
          'short_title' => Text::_('COM_FORMEA_ELEMENT_TYPES'),
          'link'        => Route::_('index.php?option=com_formea&view=eltypes'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'fas fa-palette',
          'title'       => Text::_('COM_FORMEA_THEMES_MANAGE'),
          'short_title' => Text::_('COM_FORMEA_THEMES_MANAGE'),
          'link'        => Route::_('index.php?option=com_formea&view=themes'),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'fas fa-project-diagram',
          'title'       => Text::_('COM_FORMEA_CORES_MANAGER'),
          'short_title' => Text::_('COM_FORMEA_CORES_MANAGER'),
          'link'        => Route::_('index.php?option=com_formea&view=coreextensions'),
          'internal'    => true,
        ],
      ];
      // http://localhost:8016/administrator/index.php?option=com_config&view=component&component=com_formea&path=&return=aHR0cDovL2xvY2FsaG9zdDo4MDE2L2FkbWluaXN0cmF0b3IvaW5kZXgucGhwP29wdGlvbj1jb21fZm9ybWVhJnZpZXc9ZGFzaGJvYXJk
      $uri         = (string) Uri::getInstance();
      $return      = urlencode(base64_encode($uri));
      $systemItems = [
        (object) [
          'icon'        => 'fas fa-cogs',
          'title'       => Text::_('COM_FORMEA_SETTINGS'),
          'short_title' => Text::_('COM_FORMEA_SETTINGS'),
          'link'        => Route::_('?option=com_config&view=component&component=com_formea&path=&return=' . $return),
          'internal'    => true,
        ],
        (object) [
          'icon'        => 'fas fa-book',
          'title'       => Text::_('COM_FORMEA_DOCUMENTATION'),
          'short_title' => Text::_('COM_FORMEA_DOCUMENTATION'),
          'link'        => FORMEA_DOC_LINK,
          'internal'    => false,
        ],
        (object) [
          'icon'        => 'far fa-life-ring',
          'title'       => Text::_('COM_FORMEA_SUPPORT'),
          'short_title' => Text::_('COM_FORMEA_SUPPORT'),
          'link'        => FORMEA_SUPPORT_LINK,
          'internal'    => false,
        ],
      ];
      if (empty($section))
      {
        $ret = array_merge($ret, $manageItems);
        $ret = array_merge($ret, $installableItems);
        $ret = array_merge($ret, $systemItems);
      }
      else
      {
        if (in_array('MANAGE'))
        {
          $ret = array_merge($ret, $manageItems);
        }
        if (in_array('INSTALL'))
        {
          $ret = array_merge($ret, $installableItems);
        }
        if (in_array('SYSTEM'))
        {
          $ret = array_merge($ret, $systemItems);
        }
      }

      return $ret;
    }

    /**
     * Get list of Eltypes
     *
     * @param   int  $state
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function getInstalledElTypes($state = -1)
    {
      $container = Factory::getContainer();
      $db        = $container->get('DatabaseDriver');
      $query     = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_eltypes'));
      if ($state > -1)
      {
        $query->where($db->quoteName('state') . ' = ' . $db->quote($state));
      }
      $query->order('id ASC');
      $db->setQuery($query);
      $results = $db->loadObjectList();
      if (!empty($results))
      {
        $elementPath  = JPATH_ROOT . '/components/com_formea/plugins/element/';
        $totalResults = count($results);
        for ($i = 0; $i < $totalResults; $i++)
        {
          if (is_file(Path::clean($elementPath . $results[$i]->name . '/element.json')))
          {
            $elementJson         = file_get_contents($elementPath . $results[$i]->name . '/element.json');
            $results[$i]->config = new Registry(json_decode($elementJson));
          }
          else
          {
            $results[$i]->config = new Registry((object) []);
          }
        }
      }

      return $results;
    }

    /**
     * Get list of installed validation plugins
     *
     * @param   int  $state
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function getInstalledValidations($state = -1)
    {
      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_validations'));
      if ($state > -1)
      {
        $query->where($db->quoteName('state') . ' = ' . $db->quote($state));
      }
      $query->order('id ASC');
      $db->setQuery($query);
      $results = $db->loadObjectList();
      if (!empty($results))
      {
        $elementPath  = JPATH_ROOT . '/components/com_formea/plugins/validation/';
        $totalResults = count($results);
        for ($i = 0; $i < $totalResults; $i++)
        {
          if (is_file(Path::clean($elementPath . $results[$i]->name . '/validation.json')))
          {
            $elementJson         = file_get_contents($elementPath . $results[$i]->name . '/validation.json');
            $results[$i]->config = new Registry(json_decode($elementJson));
          }
          else
          {
            $results[$i]->config = new Registry((object) []);
          }
        }
      }

      return $results;
    }

    /**
     * Get list of installed themes plugins
     *
     * @param   int  $state
     *
     * @return array
     *
     * @since 1.0.0
     */
    public static function getInstalledThemes($state = -1)
    {
      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_themes'));
      if ($state > -1)
      {
        $query->where($db->quoteName('state') . ' = ' . $db->quote($state));
      }
      $query->order('id ASC');
      $db->setQuery($query);
      $results = $db->loadObjectList();
      if (!empty($results))
      {

        $totalResults = count($results);
        for ($i = 0; $i < $totalResults; $i++)
        {
          if (is_file(Path::clean(FORMEA_THEME_PATH . '/' . $results[$i]->name . '/validation.json')))
          {
            $elementJson         = file_get_contents(FORMEA_THEME_PATH . '/' . $results[$i]->name . '/theme.json');
            $results[$i]->config = new Registry(json_decode($elementJson));
          }
          else
          {
            $results[$i]->config = new Registry((object) []);
          }
        }
      }

      return $results;
    }

    /**
     * Method to get the documentation link based on view
     *
     * @param $view
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function getHelpLink($view)
    {
      $link   = FORMEA_DOC_LINK;
      $_views = [
        'Dashboard'       => 'getting-started',
        'Element'         => 'elements',
        'Elements'        => 'elements',
        'Eltype'          => 'installable/element-types',
        'Eltypes'         => 'installable/element-types',
        'Formea'          => 'forms',
        'Formeas'         => 'forms',
        'Themes'          => 'installable/themes',
        'Theme'           => 'installable/themes',
        'Submission'      => 'submissions',
        'Submissions'     => 'submissions',
        'GroupedElements' => 'grouped-element',
        'GroupedElement'  => 'grouped-element',
        'Textbox'         => 'installable/element-types/element-lists/inputbox-field',
        'Validation'      => 'installable/validations',
        'Validations'     => 'installable/validations',
      ];
      if (isset($_views[$view]))
      {
        $link .= '/' . $_views[$view];
      }

      return $link;
    }

    public static function formatOptionTextValModel($options = [])
    {
      $ret = [];
      foreach ($options as $langCode => $data)
      {
        $data['option_lists'] = json_encode($data['option_lists']);
        $ret[$langCode]       = $data;
      }

      return $ret;
    }

    public static function generateExtensionStyleXml($dynamicFields)
    {
      $con = '<field
    name="styles"
    type="subform"
    label="COM_FORMEA_STYLES"
    description="COM_FORMEA_STYLES"
    multiple="true"
    layout="joomla.form.field.subform.repeatable"
  >
    <form
      addruleprefix="Joomla\Component\Formea\Administrator\Rule"
      addfieldprefix="Joomla\Component\Formea\Administrator\Field"
    >';
      $con .= $dynamicFields;
      $con .= ' </form>
  </field>';

      return $con;
    }


    public static function getCommonGridItems($is_preview = false, $elementsWhere = [], $groupWhere = [], $db_limit = [], $ordering = ['created_date' => 'DESC'])
    {
      $limit  = 5;
      $offset = 0;
      if (!empty($db_limit))
      {
        if (isset($db_limit['limit']))
        {
          $limit = (int) $db_limit['limit'];
        }
        if (isset($db_limit['offset']))
        {
          $offset = (int) $db_limit['offset'];
        }
      }


      $elementColumns = [
        'a.id AS id',
        'a.id AS element_id',
        'a.type',
        'a.title',
        'a.alias',
        'a.state',
        'b.title AS elementTypeTitle',
        'b.value_type',
        'b.id AS eltype_id',
        '0 AS isGroup',
        '0 AS group_id',
        'a.created_date',
      ];
      $container      = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select($elementColumns);
      $query->from($db->quoteName('#__formea_elements', 'a'))->innerJoin(
        $db->quoteName('#__formea_eltypes', 'b') . ' ON ' . $db->quoteName('a.type') . ' = ' .
        $db->quoteName('b.name')
      );

      if (!empty($elementsWhere))
      {
        foreach ($elementsWhere as $clause)
        {
          $v = $db->quote($clause->value);
          if ($clause->value[0] == '(' && $clause->value[strlen($clause->value) - 1] == ')')
          {
            $v = $clause->value;
          }
          $query->where(
            $db->quoteName($clause->field) . ' ' .
            $clause->operator . ' ' .
            $v
          );
        }
      }


      $groupedText = Text::_('COM_FORMEA_GROUPED_ELEMENTS');
      //group queries
      $groupColumns = [
        'a.id AS id',
        '-1 AS element_id',
        '\'' . $groupedText . '\' AS type',
        'a.title',
        'a.alias',
        'a.state',
        '\'' . $groupedText . '\' AS elementTypeTitle',
        '-1 AS value_type',
        '-1 AS eltype_id',
        '1 AS isGroup',
        'a.id AS group_id',
        'a.created_date',
      ];
      $qroupQuery   = $db->getQuery(true);
      $qroupQuery->select($groupColumns);
      $qroupQuery->from($db->quoteName('#__formea_groups', 'a'));
      if (!empty($groupWhere))
      {
        foreach ($groupWhere as $clause)
        {
          $v = $db->quote($clause->value);
          if ($clause->value[0] == '(' && $clause->value[strlen($clause->value) - 1] == ')')
          {
            $v = $clause->value;
          }
          $qroupQuery->where(
            $db->quoteName($clause->field) . ' ' .
            $clause->operator . ' ' .
            $v
          );
        }
      }

      $query->unionAll($qroupQuery);
      foreach ($ordering as $column => $direction)
      {
        $query->order($db->quoteName($column) . ' ' . $direction);
      }

      if ($limit > 0)
      {
        $query->setLimit($limit, $offset);
      }
      $db->setQuery($query);
      $elements = $db->loadObjectList();

      $query = clone $query;
      $query->clear('limit')->clear('offset')->clear('order');
      $db->setQuery($query);
      $db->execute();
      $total = $db->getNumRows();

      $totalElements = count($elements);
      for ($i = 0; $i < $totalElements; $i++)
      {
        $elements[$i]->isGroup = filter_var((int) $elements[$i]->isGroup, FILTER_VALIDATE_BOOLEAN);
        $imageDetails          = [
          'image'     => '',
          'imageSafe' => '',
          'imageType' => 0
        ];

        if ($elements[$i]->isGroup)
        {
          $imageDetails                = [
            'image'     => 'media/com_formea/img/grouped-elements.svg',
            'imageSafe' => '',
            'imageType' => 1
          ];
          $elements[$i]->labelSettings = [];
          $formeaGroupedElement        = new FormeaGroupedElement($elements[$i]->group_id, ['isPreview' => $is_preview]);
          $inputGroupItem              = $formeaGroupedElement->getInputGroupParams();
          $rows                        = $formeaGroupedElement->getLayout();
          $groupContent                = [
            'group_type'     => $formeaGroupedElement->group_type,
            'rows'           => $rows,
            'inputGroupItem' => $inputGroupItem
          ];
          $elements[$i]->groupContent  = $groupContent;
        }
        else
        {
          $_element                       = new FormeaElement(['element_id' => $elements[$i]->element_id, 'is_preview' => $is_preview]);
          $_elementTypeTitle              = $_element->getElementTypeTitle();
          $elements[$i]->labelSettings    = $_element->getLabelSettings();
          $imageDetails                   = $_element->getElementTypeImage();
          $elements[$i]->groupContent     = [];
          $elements[$i]->elementTypeTitle = $_elementTypeTitle;
        }
        if ($imageDetails['imageType'] === 1)
        {
          if (is_file(Path::clean(JPATH_ROOT . '/' . $imageDetails['image'])))
          {
            $imageDetails['image'] = file_get_contents(JPATH_ROOT . '/' . $imageDetails['image']);
          }
        }
        else
        {
          $imageDetails['image'] = Uri::base() . $imageDetails['image'];
        }
        $elements[$i]->imageDetails = $imageDetails;
        $elements[$i]->used         = false;
      }

      $page = 1;
      if ($offset > 0)
      {
        $page = ceil(($offset + 1) / $limit);
      }
      $fromResult = $offset + 1;
      $toResult   = 0;
      if ($offset + $limit < $total)
      {
        $toResult = $offset + $limit;
      }
      else
      {
        $toResult = $total;
      }
      $ret = [
        'pagination' => [
          'total'      => $total,
          'limit'      => $limit,
          'offset'     => $offset,
          'page'       => (int) $page,
          'fromResult' => $fromResult,
          'toResult'   => $toResult
        ],
        'items'      => $elements,
      ];

      return $ret;
    }

    /**
     * Method to get the email status from submission
     *
     * @since 1.2.0
     * @return stdClass
     */
    public static function extractSubmissionEmailStatus($submissionData)
    {
      $ret = [
        'adminBg'   => 'text-bg-secondary',
        'adminIcon' => 'fa fa-minus',
        'adminText' => Text::_('COM_FORMEA_EMAIL_NO_ATTEMPT'),
        'userBg'   => 'bg-secondary',
        'userIcon' => 'fa fa-minus',
        'userText' => Text::_('COM_FORMEA_EMAIL_NO_ATTEMPT'),
      ];

      if(!empty($submissionData->admin_email_notes)){
        $adminNotes = json_decode($submissionData->admin_email_notes);
        if((int)$submissionData->admin_email_sent > 0){
          $ret['adminText'] = Text::_('COM_FORMEA_EMAIL_SENT_WITH_FOLLOWING_ERRORS').'<br/>- ';
          $ret['adminText'] .= implode('<br/>- ',$adminNotes);
          $ret['adminBg']='text-bg-warning';
          $ret['adminIcon']='fas fa-exclamation-triangle';
        }elseif ((int)$submissionData->admin_email_sent < 1){
          $ret['adminText'] = Text::_('COM_FORMEA_EMAIL_ATTEMPT_FAIL_WITH_FOLLOWING_ERRORS').'<br/>- ';
          $ret['adminText'] .= implode('<br/>- ',$adminNotes);
          //fail
          $ret['adminBg']='text-bg-danger';
          $ret['adminIcon']='fas fa-ban';
        }
      }elseif ((int)$submissionData->admin_email_sent > 0){
        $ret['adminText'] = Text::sprintf('COM_FORMEA_EMAIL_SUCCESSFULLY_SENT_TO_S',$submissionData->admin_email);
        $ret['adminBg']='text-bg-success';
        $ret['adminIcon']='fas fa-check';
      }

      if(!empty($submissionData->user_email_notes)){
        $userNotes = json_decode($submissionData->user_email_notes);
        if((int)$submissionData->user_email_sent > 0){
          $ret['userText'] = Text::_('COM_FORMEA_EMAIL_SENT_WITH_FOLLOWING_ERRORS').'<br/>- ';
          $ret['userText'] .= implode('<br/>- ',$userNotes);
          $ret['userBg']='text-bg-warning';
          $ret['userIcon']='fas fa-exclamation-triangle';
        }elseif ((int)$submissionData->user_email_sent < 1){
          $ret['userText'] = Text::_('COM_FORMEA_EMAIL_ATTEMPT_FAIL_WITH_FOLLOWING_ERRORS').'<br/>- ';
          $ret['userText'] .= implode('<br/>- ',$userNotes);
          //fail
          $ret['userBg']='text-bg-danger';
          $ret['userIcon']='fas fa-ban';
        }
      }elseif ((int)$submissionData->user_email_sent > 0){
        $ret['userText'] = Text::sprintf('COM_FORMEA_EMAIL_SUCCESSFULLY_SENT_TO_S',$submissionData->user_email);
        $ret['userBg']='text-bg-success';
        $ret['userIcon']='fas fa-check';
      }
      return (object)$ret;
    }


  }
