<?php
  /**
   * @package     Joomla.Administrator
   * @subpackage  com_formea
   *
   * @copyright   Copyright (C) 2010-2024 Feseur Sdn Bhd. All rights reserved.
   * @license     GNU General Public License version 2 or later; see LICENSE.txt
   * @version     1.2.2
	 */

  namespace Joomla\Component\Formea\Administrator\Model;

  defined('_JEXEC') or die;

  use Exception;
  use Feseur\Library\FsrDate;
  use Feseur\Library\FsrHelper;
  use Feseur\Library\FsrResponse;
  use Joomla\CMS\Application\ApplicationHelper;
  use Joomla\CMS\Event\Model\BeforeBatchEvent;
  use Joomla\CMS\Factory;
  use Joomla\CMS\Form\Form;
  use Joomla\CMS\Form\FormFactoryInterface;
  use Joomla\CMS\Language\LanguageHelper;
  use Joomla\CMS\Language\Text;
  use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
  use Joomla\CMS\MVC\Model\AdminModel;
  use Joomla\CMS\Object\CMSObject;
  use Joomla\Component\Formea\Site\Helper\FormeaGeneralHelper;
  use Joomla\Component\Formea\Site\Libraries\FormeaAttachment;
  use Joomla\Component\Formea\Site\Libraries\FormeaGridItem;
  use Joomla\Database\DatabaseDriver;
  use Joomla\Registry\Registry;
  use Joomla\Utilities\ArrayHelper;
  use SimpleXMLElement;
  use stdClass;

  /**
   * Item Model for a Formea.
   *
   * @since  1.0.0
   */
  class FormeaModel extends AdminModel
  {
    /**
     * The type alias for this content type.
     *
     * @var    string
     * @since  1.0.0
     */
    public $typeAlias = 'com_formea.formea';

    /**
     * The context used for the associations table
     *
     * @var    string
     * @since  1.0.0
     */
    protected $associationsContext = 'com_formea.item';

    /**
     * Batch copy/move command. If set to false, the batch copy/move command is not supported
     *
     * @var  string
     */
    protected $batch_copymove = 'category_id';

    /**
     * Allowed batch commands
     *
     * @var array
     */
    protected $batch_commands
      = [
        'assetgroup_id' => 'batchAccess',
        'language_id'   => 'batchLanguage',
      ];

    /**
     * @var string
     * @since 1.0.0
     */
    protected $langTag;

    /**
     * @var string
     * @since 1.0.0
     */
    protected $langTagUnderscore;

    public $hasCoreExtensions;
    public $coreExtensions;

    /**
     * The directory to the core type end with slash
     * @var string
     * @since 1.0.0
     */
    public $coreDir = '';

    public function __construct($config = array(), MVCFactoryInterface $factory = null, FormFactoryInterface $formFactory = null)
    {
      parent::__construct($config, $factory, $formFactory);
      $app                     = Factory::getApplication();
      $this->langTag           = $app->getLanguage()->getTag();
      $this->langTagUnderscore = str_replace('-', '_', $this->langTag);
      $this->coreExtensions    = FormeaGeneralHelper::getCoreExtensions(1, [0, 2]);
      if (!empty($this->coreExtensions))
      {
        $this->hasCoreExtensions = true;
      }
      else
      {
        $this->hasCoreExtensions = false;
      }
      $this->coreDir = JPATH_ROOT . '/components/com_formea/plugins/core/';
    }

    /**
     * Method to get the row form.
     *
     * @param   array    $data      Data for the form.
     * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
     *
     * @return  Form|boolean  A \Form object on success, false on failure
     *
     * @throws  Exception
     * @since   1.0.0
     */
    public function getForm($data = array(), $loadData = true)
    {
      $app     = Factory::getApplication();
      $xml     = file_get_contents(JPATH_ADMINISTRATOR . '/components/com_formea/forms/formea.xml');
      $langTag = $app->getLanguage()->getTag();
      $con     = '<fields name="details">' . "\r\n";
      $con     .= $this->_addLangFields($langTag);
      $pk      = (!empty($pk)) ? $pk : (int) $this->getState($this->getName() . '.id');
      /*
       * Submission translatable
       * <field name="sub_message"/>
      */
      $sub_message = '<fields name="submission">' . "\r\n";
      $sub_message .= $this->_addLangSubmissionFields($langTag);

      $email_content = '<fields name="contents">' . "\r\n";
      $email_content .= $this->_addLangEmailFields($langTag, $pk);

      $languages      = FormeaGeneralHelper::getLanguages();
      $totalLanguages = count($languages);
      for ($i = 0; $i < $totalLanguages; $i++)
      {
        if ($languages[$i]->lang_code !== $langTag)
        {
          $con           .= $this->_addLangFields($languages[$i]->lang_code);
          $sub_message   .= $this->_addLangSubmissionFields($languages[$i]->lang_code);
          $email_content .= $this->_addLangEmailFields($languages[$i]->lang_code, $pk);
        }
      }
      $con .= '</fields>';

      $sub_message   .= '</fields>';
      $email_content .= '</fields>';

      //check for core extensions
      $extensionXml    = ' <fields name="cores">' . "\r\n";
      $hasExtensionXML = false;

      $replaceables = [
        'FORMID="0"' => 'data-formid="' . $pk . '"'
      ];
      if ($this->hasCoreExtensions)
      {

        $totalExtensions = count($this->coreExtensions);
        for ($j = 0; $j < $totalExtensions; $j++)
        {
          $filePathXml = $this->coreDir . $this->coreExtensions[$j]->name . '/' . $this->coreExtensions[$j]->name . '.xml';
          $root        = simplexml_load_file($filePathXml);
          if (isset($root->formConfig))
          {
            $configField = $root->formConfig->children()->asXML();
            if (!empty($configField))
            {
              foreach ($replaceables as $rk => $rv)
              {
                $configField = str_replace($rk, $rv, $configField);
              }
              $extensionXml    .= $configField;
              $hasExtensionXML = true;
            }
          }
        }
      }
      $extensionXml .= '</fields>';


      $xml = str_replace('<fields name="details"/>', $con, $xml);
      $xml = str_replace('<fields name="submission"/>', $sub_message, $xml);
      $xml = str_replace('<fields name="admin_contents"/>', $email_content, $xml);
      $xml = str_replace('<fields name="user_contents"/>', $email_content, $xml);
      if ($hasExtensionXML)
      {
        $xml = str_replace('<fields name="cores"/>', $extensionXml, $xml);
      }

      //<fields name="admin_contents"/>
      //<fields name="details"/>

      // Get the form.
      $form = $this->loadForm('com_formea.formea',
        $xml, ['control' => 'jform', 'load_data' => $loadData]);

      if (empty($form))
      {
        return false;
      }

      return $form;
    }

    protected function _addLangFields($lang_code)
    {
      $lang_code = str_replace('-', '_', $lang_code);
      $con       = '    <fields name="' . $lang_code . '">' . "\r\n";
      $con       .= '      <field name="form_heading"  label="COM_FORMEA_FORM_HEADING" type="text"/>' . "\r\n";
      $con       .= '      <field name="metakey" type="textarea" label="JFIELD_META_KEYWORDS_LABEL" rows="3" cols="30" /> ';
      $con       .= '      <field name="metadesc" type="textarea" label="JFIELD_META_DESCRIPTION_LABEL" rows="3" cols="30" maxlength="160" charcounter="true" />';
      $con       .= '    </fields>' . "\r\n";

      return $con;
    }

    protected function _addLangSubmissionFields($lang_code)
    {
      $lang_code = str_replace('-', '_', $lang_code);
      $con       = '    <fields name="' . $lang_code . '">' . "\r\n";
      $con       .= '      <field name="submission_msg"  label="COM_FORMEA_SUBMISSION_MSG" description="COM_FORMEA_SUBMISSION_MSG_DESC" filter="RAW" type="editor" buttons="true"/>' . "\r\n";
      $con       .= '      <field name="global_error_msg"  label="COM_FORMEA_GENERAL_ERROR_MSG" description="COM_FORMEA_GENERAL_ERROR_MSG_DESC" filter="RAW" type="editor" buttons="true"/>' . "\r\n";
      $con       .= '      <field name="limit_reach_msg"  label="COM_FORMEA_LIMIT_REACH_MSG" description="COM_FORMEA_LIMIT_REACH_MSG_DESC" filter="RAW" type="editor" buttons="true"/>' . "\r\n";
      $con       .= '    </fields>' . "\r\n";

      return $con;
    }

    protected function _addLangEmailFields($lang_code, $formId)
    {
      $lang_code = str_replace('-', '_', $lang_code);
      $con       = '    <fields name="' . $lang_code . '">' . "\r\n";
      $con       .= '      <field name="from_name"  label="COM_FORMEA_EMAIL_FROM_NAME" description="COM_FORMEA_EMAIL_FROM_NAME_DESC" type="text"/>' . "\r\n";
      $con       .= '      <field name="replyto_name"  label="COM_FORMEA_EMAIL_REPLYTO_NAME" description="COM_FORMEA_EMAIL_REPLYTO_NAME_DESC" type="text"/>' . "\r\n";
      $con       .= '      <field name="subject"  label="COM_FORMEA_EMAIL_SUBJECT"  type="text" />' . "\r\n";
      $con       .= '      <field name="email_content"  label="COM_FORMEA_EMAIL_CONTENT" type="editor" filter="raw" buttons="true"/>' . "\r\n";
      $con       .= '      <field name="attachments" form-id="' . $formId . '"  label="COM_FORMEA_ATTACHMENTS" type="FormeaAttachments" />' . "\r\n";
      $con       .= '    </fields>' . "\r\n";

      return $con;
    }

    /**
     * Method to get the data that should be injected in the form.
     *
     * @return  mixed  The data for the form.
     *
     * @throws  Exception
     * @since   1.0.0
     */
    protected function loadFormData()
    {

      $data = $this->getItem();

      $this->preprocessData('com_formea.formea', $data);

      return $data;
    }

    /**
     * Method to get a single record.
     *
     * @param   integer  $pk  The id of the primary key.
     *
     * @return  mixed  Object on success, false on failure.
     *
     * @throws  Exception
     * @since   1.0.0
     */
    public function getItem($pk = null)
    {
      $pk    = (!empty($pk)) ? $pk : (int) $this->getState($this->getName() . '.id');
      $table = $this->getTable();

      if ($pk > 0)
      {
        // Attempt to load the row.
        $return = $table->load($pk);

        // Check for a table object error.
        if ($return === false)
        {
          // If there was no underlying error, then the false means there simply was not a row in the db for this $pk.
          if (!$table->getError())
          {
            $this->setError(Text::_('JLIB_APPLICATION_ERROR_NOT_EXIST'));
          }
          else
          {
            $this->setError($table->getError());
          }

          return false;
        }
      }

      // Convert to the CMSObject before adding other data.
      $properties  = $table->getProperties(1);
      $detailArray = [];
      $themes      = [];
      $submission  = [];
      $cores       = [];
      if (empty($properties['admin_column']))
      {
        $properties['admin_column'] = '{}';
      }
      $emails = [];
      if (isset($properties['id']))
      {
        if ($properties['id'] > 0)
        {
          $container = Factory::getContainer();
          /** @var DatabaseDriver $db */
          $db    = $container->get('DatabaseDriver');
          $query = $db->getQuery(true);
          $query->select('*');
          $query->from($db->quoteName('#__formea_formdetails'));
          $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
          $db->setQuery($query);
          $details = $db->loadObjectList();
          if (!empty($details))
          {
            $totalDetails = count($details);
            for ($i = 0; $i < $totalDetails; $i++)
            {
              $lTag               = str_replace('-', '_', $details[$i]->language);
              $detailArray[$lTag] = [
                'form_heading' => $details[$i]->form_heading,
                'metakey'      => $details[$i]->metakey,
                'metadesc'     => $details[$i]->metadesc,
              ];
            }
          }

          $query->clear();
          $query->select(array('load_css', 'load_js', 'theme_id', 'style_id'));
          $query->from($db->quoteName('#__formea_formthemes'));
          $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
          $db->setQuery($query);
          $themes = $db->loadAssoc();

          $query->clear();
          $query->select(array('submission_msg', 'global_error_msg', 'limit_reach_msg', 'language'));
          $query->from($db->quoteName('#__formea_formsubmissions'));
          $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
          $db->setQuery($query);
          $submissionLists = $db->loadObjectList();
          if (!empty($submissionLists))
          {
            $totalSubmission = count($submissionLists);
            for ($i = 0; $i < $totalSubmission; $i++)
            {
              $submissionLang              = str_replace('-', '_', $submissionLists[$i]->language);
              $submission[$submissionLang] = [
                'submission_msg'   => $submissionLists[$i]->submission_msg,
                'global_error_msg' => $submissionLists[$i]->global_error_msg,
                'limit_reach_msg'  => $submissionLists[$i]->limit_reach_msg,
              ];
            }
          }

          $query->clear();
          $query->select(array('id', 'target_type', 'from_email', 'to_email', 'cc_email', 'bcc_email', 'replyto_email'));
          $query->from($db->quoteName('#__formea_formemails'));
          $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
          $db->setQuery($query);
          $emailLists = $db->loadObjectList();
          if (!empty($emailLists))
          {
            $totalEmails = count($emailLists);
            for ($i = 0; $i < $totalEmails; $i++)
            {
              //get content
              $query->clear();
              $query->select(array('from_name', 'replyto_name', 'subject', 'email_content', 'language', 'attachments'));
              $query->from($db->quoteName('#__formea_formemailcontent'));
              $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
              $query->where($db->quoteName('email_id') . ' = ' . $emailLists[$i]->id);
              $db->setQuery($query);
              $contentLists = $db->loadObjectList();
              $content      = [];
              if (!empty($contentLists))
              {
                $totalContents = count($contentLists);
                for ($j = 0; $j < $totalContents; $j++)
                {
                  $contentLang           = str_replace('-', '_', $contentLists[$j]->language);
                  $content[$contentLang] = [
                    'from_name'     => $contentLists[$j]->from_name,
                    'replyto_name'  => $contentLists[$j]->replyto_name,
                    'subject'       => $contentLists[$j]->subject,
                    'email_content' => $contentLists[$j]->email_content,
                    'attachments'   => $emailLists[$i]->id . '|#|' . $contentLists[$j]->language
                  ];
                }
              }
              switch ($emailLists[$i]->target_type)
              {
                case 1:
                  $emailTgtType = 'user';
                  break;
                case 2:
                  $emailTgtType = 'others';
                  break;
                default:
                  $emailTgtType = 'admin';
              }

              $emails[$emailTgtType] = [
                'from_email'    => $emailLists[$i]->from_email,
                'to_email'      => $emailLists[$i]->to_email,
                'cc_email'      => $emailLists[$i]->cc_email,
                'bcc_email'     => $emailLists[$i]->bcc_email,
                'replyto_email' => $emailLists[$i]->replyto_email,
                'contents'      => $content
              ];
            }
          }

          $query->clear();
          $query->select('*');
          $query->from($db->quoteName('#__formea_form_cores'));
          $query->where($db->quoteName('form_id') . ' = ' . $properties['id']);
          $db->setQuery($query);
          $coreLists      = $db->loadObjectList();
          $totalCoreLists = count($coreLists);
          for ($k = 0; $k < $totalCoreLists; $k++)
          {
            $coreParams = json_decode($coreLists[$k]->params);
            foreach ($coreParams as $cpKey => $cpVal)
            {
              $cores[$coreLists[$k]->corename][$cpKey] = $cpVal;
            }
          }
        }
      }
      $properties['details']       = $detailArray;
      $properties['themes']        = $themes;
      $properties['submission']    = $submission;
      $properties['emails']        = $emails;
      $properties['elementlayout'] = (isset($properties['id'])) ? $properties['id'] : 0;
      $properties['cores']         = $cores;

      $item = ArrayHelper::toObject($properties, CMSObject::class);

      if (property_exists($item, 'params'))
      {
        $registry     = new Registry($item->params);
        $item->params = $registry->toArray();
      }

      return $item;
    }


    public function save($data)
    {
      $app         = Factory::getApplication();
      $input       = $app->getInput();
      $user        = $app->getIdentity();
      $isNew       = false;
      $currentDate = new FsrDate();
      if ((int) $data['id'] === 0)
      {
        $data['created_date']  = $currentDate->toSql();
        $data['created_by']    = $user->id;
        $data['modified_date'] = $currentDate->toSql();
        $data['modified_by']   = $user->id;
        $isNew                 = true;
      }
      else
      {
        $data['modified_date'] = $currentDate->toSql();
        $data['modified_by']   = $user->id;
      }
      if (isset($data['admin_column']))
      {
        $data['admin_column'] = json_encode($data['admin_column']);
      }
      if ($input->get('task') == 'save2copy')
      {
        $data['title'] = $data['title'] . ' - ' . Text::_('COM_FORMEA_COPY');
        $data['alias'] = ApplicationHelper::stringURLSafe($data['title']);
      }

      $saved   = parent::save($data);
      $form_id = $this->getState($this->getName() . '.id');

      $this->_storeFormDetails((int) $form_id, $data['details'], $isNew);
      $this->_storeFormThemes((int) $form_id, $data['themes'], $isNew);
      $this->_storeFormEmailData((int) $form_id, $data['emails'], $isNew);
      $this->_storeFormSubmissionData((int) $form_id, $data['submission'], $isNew);
      if (isset($data['elementlayout']))
      {
        $elementLayout = json_decode($data['elementlayout']);
        $this->_storeFormLayout((int) $form_id, $elementLayout, $isNew);
      }

      //store cores
      if (isset($data['cores']))
      {
        $this->_storeCoreExtensions((int) $form_id, $data['cores'], $isNew);
      }

      return $saved;
    }

    public function _storeCoreExtensions($form_id, $details, $isNew = false)
    {
      if ($form_id > 0)
      {
        $db = Factory::getDbo();
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query      = $db->getQuery(true);
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          $query->delete($db->quoteName('#__formea_form_cores'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
        foreach ($details as $corename => $coreParam)
        {
          $enabled = 0;
          if (isset($coreParam['enabled']))
          {
            $enabled = (int) $coreParam['enabled'];
          }
          $obj = (object) [
            'form_id'  => $form_id,
            'enabled'  => $enabled,
            'corename' => $corename,
            'params'   => json_encode($coreParam)
          ];
          $db->insertObject('#__formea_form_cores', $obj);
        }
      }

      return true;
    }

    public function _storeFormDetails($form_id, $details, $isNew = false)
    {
      if ($form_id > 0)
      {
        $db = Factory::getDbo();
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query      = $db->getQuery(true);
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          $query->delete($db->quoteName('#__formea_formdetails'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
        foreach ($details as $lang_code => $detailData)
        {
          $obj               = new stdClass();
          $obj->form_id      = $form_id;
          $obj->form_heading = $detailData['form_heading'];
          $obj->metakey      = $detailData['metakey'];
          $obj->metadesc     = $detailData['metadesc'];
          $obj->language     = str_replace('_', '-', $lang_code);
          $db->insertObject('#__formea_formdetails', $obj);
        }
      }

      return true;
    }

    public function _storeFormSubmissionData($form_id, $details, $isNew = false)
    {
      if ($form_id > 0)
      {
        $db = Factory::getDbo();
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query      = $db->getQuery(true);
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          $query->delete($db->quoteName('#__formea_formsubmissions'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
        $defaultValue = FormeaGeneralHelper::getDefaultLangValField([
          'submission_msg',
          'global_error_msg',
          'limit_reach_msg'
        ], $details, $this->langTagUnderscore);
        foreach ($details as $lang_code => $detailData)
        {
          $submission_msg   = $detailData['submission_msg'];
          $global_error_msg = $detailData['global_error_msg'];
          $limit_reach_msg  = $detailData['limit_reach_msg'];
          if (empty($submission_msg) && !empty($defaultValue))
          {
            $submission_msg = $defaultValue['submission_msg'];
          }
          if (empty($global_error_msg) && !empty($defaultValue))
          {
            $global_error_msg = $defaultValue['global_error_msg'];
          }
          if (empty($limit_reach_msg) && !empty($defaultValue))
          {
            $limit_reach_msg = $defaultValue['limit_reach_msg'];
          }
          $obj                   = new stdClass();
          $obj->form_id          = $form_id;
          $obj->submission_msg   = $submission_msg;
          $obj->global_error_msg = $global_error_msg;
          $obj->limit_reach_msg  = $limit_reach_msg;
          $obj->language         = str_replace('_', '-', $lang_code);
          $db->insertObject('#__formea_formsubmissions', $obj);
        }
      }

      return true;
    }

    public function _storeFormThemes($form_id, $themeData, $isNew = false)
    {
      if ($form_id > 0)
      {
        $db = Factory::getDbo();
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query      = $db->getQuery(true);
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          $query->delete($db->quoteName('#__formea_formthemes'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
        $obj           = new stdClass();
        $obj->form_id  = $form_id;
        $obj->theme_id = $themeData['theme_id'];
        $obj->style_id = $themeData['style_id'];
        $obj->load_css = $themeData['load_css'];
        $obj->load_js  = $themeData['load_js'];
        $db->insertObject('#__formea_formthemes', $obj);
      }

      return true;
    }


    public function _storeFormLayout($form_id, array $layoutData, $isNew = false)
    {
      if ($form_id > 0)
      {
        $container = Factory::getContainer();
        $db        = $container->get('DatabaseDriver');
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query             = $db->getQuery(true);
          $targetTables      = [
            ['table' => 'formea_form_fieldsets', 'key' => null],
            ['table' => 'formea_form_rows', 'key' => 'row_type'],
            ['table' => 'formea_form_columns', 'key' => 'column_type'],
            ['table' => 'formea_form_elements', 'key' => 'target_type'],
            ['table' => 'formea_form_groups', 'key' => 'group_type'],
          ];
          $totalTargetTables = count($targetTables);
          for ($j = 0; $j < $totalTargetTables; $j++)
          {
            $conditions = array(
              $db->quoteName('form_id') . ' = ' . $form_id,
            );
            if (!empty($targetTables[$j]['key']))
            {
              $conditions[] = $db->quoteName($targetTables[$j]['key']) . ' = 0';
            }
            $query->clear();
            $query->delete($db->quoteName('#__' . $targetTables[$j]['table']));
            $query->where($conditions);
            $db->setQuery($query);
            $db->execute();
          }
        }
        $totalElement = count($layoutData);
        for ($i = 0; $i < $totalElement; $i++)
        {
          $formPage = (object) [
            'form_id'  => $form_id,
            'title'    => $layoutData[$i]->title,
            'subtitle' => $layoutData[$i]->subtitle,
            'icon'     => $layoutData[$i]->icon,
            'settings' => json_encode([]),
          ];
          $db->insertObject('#__formea_form_fieldsets', $formPage);
          $formPage->id     = $db->insertid();
          $styleSheetParams = FormeaGeneralHelper::getStyleSheetParams();
          $structure        = $this->_storeFormeaGridItem($form_id, $formPage->id, $layoutData[$i]->rows, $styleSheetParams);
          $upd              = new stdClass();
          $upd->id          = $formPage->id;
          $upd->layout      = json_encode($structure);
          $db->updateObject('#__formea_form_fieldsets', $upd, 'id');
        }
      }

      return true;
    }

    protected function _storeFormeaGridItem($form_id, $formPageId, $items, $styleSheetParams)
    {
      $structure  = [];
      $totalItems = count($items);
      $db         = $this->getDatabase();
      for ($i = 0; $i < $totalItems; $i++)
      {
        $targetId  = 0;
        $item_type = $items[$i]->item_type;
        $settings  = json_encode($items[$i]->settings);
        $set       = [];
        if ($items[$i]->item_type == 3)
        {
          $table = '#__formea_form_groups';
          //its a grouped element
          $groupObject = [
            'form_id'    => $form_id,
            'page_id'    => $formPageId,
            'group_id'   => $items[$i]->group_id,
            'group_type' => 0,
          ];
          $groupObject = (object) $groupObject;
          $db->insertObject($table, $groupObject);
          $groupObject->id = $db->insertid();
          $items[$i]->id   = $groupObject->id;
          // $styleSheet        = $this->_processStyleSheet($group_id, $settings, $items[$i], $styleSheetParams);
          $targetId = $groupObject->id;
          $groupId  = $items[$i]->group_id;
          $setPush  = [
            'form_group_id' => $targetId,
            'group_type'    => $items[$i]->group_type,
            'rows'          => []
          ];
          if ($items[$i]->group_type == 1)
          {
            //input group, store elements
            $totalElements = count($items[$i]->sets[0]->elements);
            for ($j = 0; $j < $totalElements; $j++)
            {
              //its an element
              $_elementObject = [
                'form_id'          => $form_id,
                'page_id'          => $formPageId,
                'group_id'         => $groupId,
                'settings'         => $settings,
                'element_id'       => $items[$i]->sets[0]->elements[$j]->element_id,
                'title'            => $items[$i]->sets[0]->elements[$j]->title,
                'elementTypeTitle' => $items[$i]->sets[0]->elements[$j]->elementTypeTitle,
                'type'             => $items[$i]->sets[0]->elements[$j]->type,
                'labelSettings'    => json_encode($items[$i]->sets[0]->elements[$j]->labelSettings),
              ];
              $_elementObject = (object) $_elementObject;
              $db->insertObject('#__formea_form_elements', $_elementObject);
            }
          }
          else
          {
            $rows            = $this->_storeFormeaGridItem($form_id, $formPageId, $items[$i]->sets[0]->rows, $styleSheetParams);
            $setPush['rows'] = $rows;
          }
          $set[] = $setPush;
        }
        else if ($items[$i]->item_type == 0)
        {
          $table = '#__formea_form_rows';
          //its a row
          $rowObject = [
            'form_id'  => $form_id,
            'page_id'  => $formPageId,
            'group_id' => $items[$i]->group_id,
            'settings' => $settings,
            'class'    => $items[$i]->class,
            'gutterX'  => $items[$i]->gutterX,
            'gutterY'  => $items[$i]->gutterY,
          ];
          $rowObject = (object) $rowObject;
          $db->insertObject($table, $rowObject);
          $rowObject->id = $db->insertid();
          $items[$i]->id = $rowObject->id;
          $styleSheet    = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
          $targetId      = $rowObject->id;
        }
        elseif ($items[$i]->item_type == 1)
        {
          $table = '#__formea_form_columns';
          //its a column
          $columnObject = [
            'form_id'  => $form_id,
            'page_id'  => $formPageId,
            'group_id' => $items[$i]->group_id,
            'settings' => $settings,
            'class'    => $items[$i]->class,
            'column'   => $items[$i]->column,
          ];
          $columnObject = (object) $columnObject;
          $db->insertObject($table, $columnObject);
          $columnObject->id = $db->insertid();
          $items[$i]->id    = $columnObject->id;
          $styleSheet       = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
          $targetId         = $columnObject->id;
        }
        elseif ($items[$i]->item_type == 2)
        {
          $table = '#__formea_form_elements';
          //its an element
          $elementObject = [
            'form_id'          => $form_id,
            'page_id'          => $formPageId,
            'group_id'         => $items[$i]->group_id,
            'settings'         => $settings,
            'element_id'       => $items[$i]->content->element_id,
            'title'            => $items[$i]->content->title,
            'elementTypeTitle' => $items[$i]->content->elementTypeTitle,
            'type'             => $items[$i]->content->type,
            'labelSettings'    => json_encode($items[$i]->content->labelSettings),
          ];
          $elementObject = (object) $elementObject;
          $db->insertObject($table, $elementObject);
          $elementObject->id = $db->insertid();
          $items[$i]->id     = $elementObject->id;
          $styleSheet        = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
          $targetId          = $elementObject->id;
        }

        $structure[$i]['item_type'] = $item_type;
        $structure[$i]['item_id']   = $targetId;
        $structure[$i]['items']     = [];
        $structure[$i]['sets']      = [];
        if (!empty($set))
        {
          $structure[$i]['sets'] = $set;
        }

        if (!empty($styleSheet) && $item_type !== 3)
        {
          $upd             = new stdClass();
          $upd->id         = $targetId;
          $upd->styleSheet = $styleSheet;
          $db->updateObject($table, $upd, 'id');
        }


        if (!empty($items[$i]->items))
        {
          $structure[$i]['items'] = $this->_storeFormeaGridItem($form_id, $formPageId, $items[$i]->items, $styleSheetParams);
        }
      }

      return $structure;
    }

    /**
     * @param   stdClass        $settings
     * @param   FormeaGridItem  $item
     * @param   array           $styleSheetParams  ;
     *
     * @return string
     *
     * @since 1.0.0
     */
    protected function _processStyleSheet($formId, $settings, $item, $styleSheetParams)
    {
      $settings = new Registry($settings);
      $elemType = 'row_' . $formId . '_' . $item->id;
      if ($item->item_type == 1)
      {
        $elemType = 'column_' . $formId . '_' . $item->id;
      }
      elseif ($item->item_type == 2)
      {
        $elemType = 'element_' . $formId . '_' . $item->content->element_id;
      }
      elseif ($item->item_type == 3)
      {
        $elemType = 'grouped_element_' . $formId . '_' . $item->group_id;
      }

      $elemSelector = '.fm_' . $elemType;
      $st           = $elemSelector . '{';
      $hasRowStyle  = false;

      foreach ($styleSheetParams as $target => $props)
      {
        $val = $settings->get($target);
        if (!empty($val))
        {
          $hasRowStyle = true;
          foreach ($props as $param)
          {
            $st .= $param . ':' . $val . ';';
          }
        }
      }
      $st .= '}';
      if ($hasRowStyle)
      {
        return $st;
      }
      else
      {
        return '';
      }
    }

    public function _storeFormEmailData($form_id, $details, $isNew = false)
    {
      if ($form_id > 0)
      {
        $db = Factory::getDbo();
        if (!$isNew)
        {
          //form id already exist, clear past details
          $query      = $db->getQuery(true);
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          $query->delete($db->quoteName('#__formea_formemails'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();

          $query->clear();
          $query->delete($db->quoteName('#__formea_formemailcontent'));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }

        $admin = [];
        $user  = [];
        if (isset($details['admin']))
        {
          $admin = $details['admin'];
        }

        if (isset($details['user']))
        {
          $user = $details['user'];
        }

        if (!empty($admin))
        {
          $this->_storeEmailData($form_id, 0, $admin);
        }
        if (!empty($user))
        {
          $this->_storeEmailData($form_id, 1, $user);
        }
      }

      return true;
    }

    protected function _storeEmailData(int $form_id, int $target_type = 0, array $admin = [])
    {
      $db                         = Factory::getDbo();
      $adminObject                = new stdClass();
      $adminObject->form_id       = $form_id;
      $adminObject->target_type   = $target_type;
      $adminObject->from_email    = $admin['from_email'];
      $adminObject->to_email      = $admin['to_email'];
      $adminObject->replyto_email = $admin['replyto_email'];
      $adminObject->cc_email      = $admin['cc_email'];
      $adminObject->bcc_email     = $admin['bcc_email'];
      $db->insertObject('#__formea_formemails', $adminObject);
      $adminObject->id   = $db->insertid();
      $adminDefaultValue = FormeaGeneralHelper::getDefaultLangValField([
        'from_name',
        'replyto_name',
        'subject',
        'email_content',
        'attachments'
      ], $admin['contents'], $this->langTagUnderscore);
      foreach ($admin['contents'] as $lang_code => $detailData)
      {
        $from_name     = $detailData['from_name'];
        $replyto_name  = $detailData['replyto_name'];
        $subject       = $detailData['subject'];
        $email_content = $detailData['email_content'];
        $attachments   = $detailData['attachments'];
        if (empty($from_name) && !empty($adminDefaultValue))
        {
          $from_name = $adminDefaultValue['from_name'];
        }
        if (empty($replyto_name) && !empty($adminDefaultValue))
        {
          $replyto_name = $adminDefaultValue['replyto_name'];
        }
        if (empty($subject) && !empty($adminDefaultValue))
        {
          $subject = $adminDefaultValue['subject'];
        }
        if (empty($email_content) && !empty($adminDefaultValue))
        {
          $email_content = $adminDefaultValue['email_content'];
        }
        $obj                = new stdClass();
        $obj->email_id      = $adminObject->id;
        $obj->form_id       = $form_id;
        $obj->from_name     = $from_name;
        $obj->replyto_name  = $replyto_name;
        $obj->subject       = $subject;
        $obj->email_content = $email_content;
        $obj->language      = str_replace('_', '-', $lang_code);
        $obj->attachments   = $attachments;
        $db->insertObject('#__formea_formemailcontent', $obj);
      }
    }

    /**
     * Method to multi delete forms
     *
     * @param   array  $pks  An array of form ids
     *
     * @return true
     *
     * @since 1.0.0
     */
    public function delete(&$pks)
    {
      if (!empty($pks))
      {
        /*
         * */
        $relatableTables = [
          ['table' => '#__formea_formdetails', 'key' => null],
          ['table' => '#__formea_formthemes', 'key' => null],
          ['table' => '#__formea_formsubmissions', 'key' => null],
          ['table' => '#__formea_submission_data', 'key' => null],
          ['table' => '#__formea_formemails', 'key' => null],
          ['table' => '#__formea_formemailcontent', 'key' => null],
          ['table' => '#__formea_form_cores', 'key' => null],
          ['table' => '#__formea_form_fieldsets', 'key' => null],
          ['table' => '#__formea_form_rows', 'key' => 'row_type'],
          ['table' => '#__formea_form_columns', 'key' => 'column_type'],
          ['table' => '#__formea_form_elements', 'key' => 'target_type'],
          ['table' => '#__formea_form_groups', 'key' => 'group_type'],
        ];
        $db              = Factory::getDbo();
        $query           = $db->getQuery(true);
        // delete all custom keys for user 1001.

        $totalTables = count($relatableTables);
        for ($i = 0; $i < $totalTables; $i++)
        {
          $query->clear();
          $conditions = [$db->quoteName('form_id') . ' IN (' . implode(',', $pks) . ')'];
          if (!empty($relatableTables[$i]['key']))
          {
            $conditions[] = $db->quoteName($relatableTables[$i]['key']) . ' = 0';
          }
          $query->delete($db->quoteName($relatableTables[$i]['table']));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }

        $conditions = array(
          $db->quoteName('id') . ' IN (' . implode(',', $pks) . ')'
        );
        $query->clear();
        $query->delete($db->quoteName('#__formea_forms'));
        $query->where($conditions);
        $db->setQuery($query);
        $db->execute();


      }

      return true;
    }


    /**
     * Method to multi delete form dependencies table
     *
     * @param   array  $pks  An array of form ids
     *
     * @return true
     *
     * @since 1.2.2
     */
    public function deleteDependencies(&$pks)
    {
      if (!empty($pks))
      {
        /*
         * */
        $relatableTables = [
          ['table' => '#__formea_formdetails', 'key' => null],
          ['table' => '#__formea_formthemes', 'key' => null],
          ['table' => '#__formea_formsubmissions', 'key' => null],
          ['table' => '#__formea_submission_data', 'key' => null],
          ['table' => '#__formea_formemails', 'key' => null],
          ['table' => '#__formea_formemailcontent', 'key' => null],
          ['table' => '#__formea_form_cores', 'key' => null],
          ['table' => '#__formea_form_fieldsets', 'key' => null],
          ['table' => '#__formea_form_rows', 'key' => 'row_type'],
          ['table' => '#__formea_form_columns', 'key' => 'column_type'],
          ['table' => '#__formea_form_elements', 'key' => 'target_type'],
          ['table' => '#__formea_form_groups', 'key' => 'group_type'],
        ];
        $db              = Factory::getDbo();
        $query           = $db->getQuery(true);
        // delete all custom keys for user 1001.

        $totalTables = count($relatableTables);
        for ($i = 0; $i < $totalTables; $i++)
        {
          $query->clear();
          $conditions = [$db->quoteName('form_id') . ' IN (' . implode(',', $pks) . ')'];
          if (!empty($relatableTables[$i]['key']))
          {
            $conditions[] = $db->quoteName($relatableTables[$i]['key']) . ' = 0';
          }
          $query->delete($db->quoteName($relatableTables[$i]['table']));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
      }

      return true;
    }

    /**
     * Method to delete form submissions associated with the form id
     *
     * @param   array  $pks  An array of form ids
     *
     * @return boolean
     * @since 1.0.0
     */
    public function deleteFormSubmissions(&$pks)
    {
      if (empty($pks))
      {
        $this->setError(Text::_('COM_FORMEA_NO_FORM_SELECTED'));

        return false;
      }

      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);

      $conditions = array(
        $db->quoteName('form_id') . ' IN (' . implode(',', $pks) . ')'
      );

      $query->delete($db->quoteName('#__formea_submissions'));
      $query->where($conditions);
      $db->setQuery($query);
      $db->execute();

      $query->clear();
      $query->delete($db->quoteName('#__formea_submission_data'));
      $query->where($conditions);
      $db->setQuery($query);
      $db->execute();

      return true;

    }

    /**
     * Method to perform batch operations on an item or a set of items.
     *
     * @param   array  $commands  An array of commands to perform.
     * @param   array  $pks       An array of item ids.
     * @param   array  $contexts  An array of item contexts.
     *
     * @return  boolean  Returns true on success, false on failure.
     *
     * @since   1.7
     */
    public function batch($commands, $pks, $contexts)
    {
      // Sanitize ids.
      $pks = array_unique($pks);
      $pks = ArrayHelper::toInteger($pks);

      // Remove any values of zero.
      if (array_search(0, $pks, true))
      {
        unset($pks[array_search(0, $pks, true)]);
      }

      if (empty($pks))
      {
        $this->setError(Text::_('JGLOBAL_NO_ITEM_SELECTED'));

        return false;
      }

      $done = false;

      // Initialize re-usable member properties
      $this->initBatch();

      if ($this->batch_copymove && !empty($commands[$this->batch_copymove]))
      {
        $cmd = ArrayHelper::getValue($commands, 'move_copy', 'c');

        if ($cmd === 'c')
        {
          $result = $this->batchCopy($commands[$this->batch_copymove], $pks, $contexts);

          if (is_array($result))
          {
            foreach ($result as $old => $new)
            {
              $contexts[$new] = $contexts[$old];
            }

            $pks = array_values($result);
          }
          else
          {
            return false;
          }
        }
        elseif ($cmd === 'm' && !$this->batchMove($commands[$this->batch_copymove], $pks, $contexts))
        {
          return false;
        }

        $done = true;
      }

      foreach ($this->batch_commands as $identifier => $command)
      {
        if (!empty($commands[$identifier]))
        {
          if (!$this->$command($commands[$identifier], $pks, $contexts))
          {
            return false;
          }

          $done = true;
        }
      }

      if (!$done)
      {
        $this->setError(Text::_('JLIB_APPLICATION_ERROR_INSUFFICIENT_BATCH_INFORMATION'));

        return false;
      }

      // Clear the cache
      $this->cleanCache();

      return true;
    }

    /**
     * Batch access level changes for a group of rows.
     *
     * @param   integer  $value     The new value matching an Asset Group ID.
     * @param   array    $pks       An array of row IDs.
     * @param   array    $contexts  An array of item contexts.
     *
     * @return  boolean  True if successful, false otherwise and internal error is set.
     *
     * @since   1.7
     */
    protected function batchAccess($value, $pks, $contexts)
    {
      // Initialize re-usable member properties, and re-usable local variables
      $this->initBatch();

      foreach ($pks as $pk)
      {
        if ($this->user->authorise('core.edit', $contexts[$pk]))
        {
          $this->table->reset();
          $this->table->load($pk);
          $this->table->access_level = (int) $value;

          $event = new BeforeBatchEvent(
            $this->event_before_batch,
            ['src' => $this->table, 'type' => 'access']
          );
          $this->dispatchEvent($event);

          // Check the row.
          if (!$this->table->check())
          {
            $this->setError($this->table->getError());

            return false;
          }

          if (!$this->table->store())
          {
            $this->setError($this->table->getError());

            return false;
          }
        }
        else
        {
          $this->setError(Text::_('JLIB_APPLICATION_ERROR_BATCH_CANNOT_EDIT'));

          return false;
        }
      }

      // Clean the cache
      $this->cleanCache();

      return true;
    }

    public function getAttachments($formId, $emailId, $langTag = 'en-GB')
    {
      $response = new FsrResponse();
      try
      {
        $fetchDb = false;
        if ($formId > 0 && $emailId > 0)
        {
          $fetchDb = true;
        }
        $app            = Factory::getApplication();
        $defaultLangTag = $app->getLanguage()->getTag();
        $languageString = FsrHelper::getCommonLanguage(true, [
          JPATH_ROOT . '/administrator/components/com_formea/language/' . $defaultLangTag . '/com_formea.ini'
        ], $langTag);
        $formeaAttachment = new FormeaAttachment();
        $iconSets = $formeaAttachment->getIconSets();

        $attachmentFiles = [];

        if($fetchDb){

          /** @var DatabaseDriver $db */
          $db = $this->getDatabase();
          $query = $db->getQuery(true);
          $query->select('attachments');
          $query->from($db->quoteName('#__formea_formemailcontent'));
          $query->where($db->quoteName('form_id') . ' = ' . $db->quote($formId));
          $query->where($db->quoteName('email_id') . ' = ' . $db->quote($emailId));
          $query->where($db->quoteName('language') . ' = ' . $db->quote($langTag));
          $db->setQuery($query);
          $result = $db->loadResult();
          if(!empty($result)){
            $filePaths = json_decode($result);
            $formeaAttachment = new FormeaAttachment();
            $attachmentFiles   = $formeaAttachment->getRelativeFiles($filePaths);
          }
        }

        $response->setSuccess(true);
        $response->setResult([
          'languageString'  => $languageString,
          'iconSets'        => $iconSets,
          'attachmentFiles' => $attachmentFiles,
        ]);
      }
      catch (Exception $e)
      {
        $response->setMsg($e->getMessage());
      }

      return $response;
    }

    public function getAttachmentFileInfo($filePaths)
    {
      $response = new FsrResponse();
      try
      {
        $processedFiles = [];
        if (!empty($filePaths))
        {
          $formeaAttachment = new FormeaAttachment();
          $processedFiles   = $formeaAttachment->getRelativeFiles($filePaths);
        }
        $response->setSuccess(true);
        $response->setResult([
          'processedFiles' => $processedFiles
        ]);
      }
      catch (Exception $e)
      {
        $response->setMsg($e->getMessage());
      }

      return $response;
    }

    public function uploadGroupedElementLayout($form_id,$layoutRows,$isNew){
      $this->_storeFormLayout($form_id, $layoutRows, $isNew);
    }

  }
