<?php
	/**
	 * @package     Joomla.Administrator
	 * @subpackage  com_formea
	 *
	 * @copyright   Copyright (C) 2010-2024 Feseur Sdn Bhd. All rights reserved.
	 * @license     GNU General Public License version 2 or later; see LICENSE.txt
	 * @version     1.2.2
	 */

	namespace Joomla\Component\Formea\Administrator\Table;

	defined('_JEXEC') or die;

	use Exception;
	use Feseur\Library\FsrDate;
	use Joomla\CMS\Application\ApplicationHelper;
	use Joomla\CMS\Factory;
	use Joomla\CMS\Language\Text;
	use Joomla\CMS\Table\Table;
	use Joomla\CMS\Tag\TaggableTableInterface;
	use Joomla\CMS\Tag\TaggableTableTrait;
	use Joomla\Database\DatabaseDriver;
	use Joomla\Registry\Registry;

	/**
	 * Formea Table class.
	 *
	 * @since  1.0.0
	 */
	class ElementTable extends Table implements TaggableTableInterface
	{
		use TaggableTableTrait;

		/**
		 * Indicates that columns fully support the NULL value in the database
		 *
		 * @var    boolean
		 * @since  1.0.0
		 */
		protected $_supportNullValue = true;

		/**
		 * Constructor
		 *
		 * @param   DatabaseDriver  $db  Database connector object
		 *
		 * @since   1.0.0
		 */
		public function __construct(DatabaseDriver $db)
		{
			$this->typeAlias = 'com_formea.element';

			parent::__construct('#__formea_elements', 'id', $db);
			$this->created_date = (new FsrDate())->toSql();
			$this->setColumnAlias('published', 'state');
		}

		/**
		 * Stores a contact.
		 *
		 * @param   boolean  $updateNulls  True to update fields even if they are null.
		 *
		 * @return  boolean  True on success, false on failure.
		 *
		 * @since   1.0.0
		 */
		public function store($updateNulls = false)
		{
			// Transform the params field
			if (is_array($this->params))
			{
				$registry     = new Registry($this->params);
				$this->params = (string) $registry;
			}

			return parent::store($updateNulls);
		}

		/**
		 * Overloaded check function
		 *
		 * @return  boolean
		 *
		 * @see     Table::check
		 * @since   1.0.0
		 */
		public function check()
		{
			try
			{
				parent::check();
			}
			catch (Exception $e)
			{
				$this->setError($e->getMessage());

				return false;
			}

			// Set name
			$this->title = htmlspecialchars_decode($this->title, ENT_QUOTES);

			// Set alias
			if (trim($this->alias) == '')
			{
				$this->alias = $this->title;
			}

			$this->alias = ApplicationHelper::stringURLSafe($this->alias);

			if (trim(str_replace('-', '', $this->alias)) == '')
			{
				$this->alias = Factory::getDate()->format('Y-m-d-H-i-s');
			}

			// Check the publish down date is not earlier than publish up.
			if ($this->publish_down > $this->_db->getNullDate() && $this->publish_down < $this->publish_up)
			{
				$this->setError(Text::_('JGLOBAL_START_PUBLISH_AFTER_FINISH'));

				return false;
			}

			// Set publish_up, publish_down to null if not set
			if (!$this->publish_up)
			{
				$this->publish_up = null;
			}

			if (!$this->publish_down)
			{
				$this->publish_down = null;
			}

			return true;
		}

		/**
		 * Get the type alias for the history table
		 *
		 * @return  string  The alias as described above
		 *
		 * @since   1.0.0
		 */
		public function getTypeAlias()
		{
			return $this->typeAlias;
		}
	}
