<?php
	/**
	 * @package     Joomla.Administrator
	 * @subpackage  com_formea
	 *
	 * @copyright   Copyright (C) 2010-2024 Feseur Sdn Bhd. All rights reserved.
	 * @license     GNU General Public License version 2 or later; see LICENSE.txt
   * @version     1.2.2
	 */

	namespace Joomla\Component\Formea\Administrator\View\Submissions;

	defined('_JEXEC') or die;

	use Exception;
	use Joomla\CMS\Application\CMSApplicationInterface;
	use Joomla\CMS\Factory;
	use Joomla\CMS\Form\Form;
	use Joomla\CMS\Helper\ContentHelper;
	use Joomla\CMS\HTML\Helpers\Sidebar;
	use Joomla\CMS\HTML\HTMLHelper;
	use Joomla\CMS\Language\Multilanguage;
	use Joomla\CMS\Language\Text;
	use Joomla\CMS\MVC\View\GenericDataException;
	use Joomla\CMS\Object\CMSObject;
	use Joomla\CMS\Pagination\Pagination;
	use Joomla\CMS\Toolbar\Toolbar;
	use Joomla\CMS\Toolbar\ToolbarHelper;
	use Joomla\Component\Formea\Administrator\Helper\FormeaHelper;
	use Joomla\Component\Formea\Administrator\Model\SubmissionsModel;
	use Joomla\Component\Formea\Administrator\View\FormeaView;
	use Joomla\Database\DatabaseDriver;
	use SimpleXMLElement;

	/**
	 * View class for a list of formeas.
	 *
	 * @since  1.0.0
	 */
	class HtmlView extends FormeaView
	{
		/**
		 * An array of items
		 *
		 * @var    array
		 * @since  1.0.0
		 */
		public $items = [];

		/**
		 * The pagination object
		 *
		 * @var    Pagination
		 * @since  1.0.0
		 */
		public $pagination;

		/**
		 * The model state
		 *
		 * @var    CMSObject
		 * @since  1.0.0
		 */
		public $state;

		/**
		 * Form object for search filters
		 *
		 * @var    Form
		 * @since  1.0.0
		 */
		public $filterForm;

		/**
		 * The active search filters
		 *
		 * @var    array
		 * @since  1.0.0
		 */
		public $activeFilters = [];

		/**
		 * The sidebar markup
		 *
		 * @var    string
		 * @since  1.0.0
		 */
		protected $sidebar;

		/**
		 * @var CMSApplicationInterface
		 * @since 1.0.0
		 */
		protected $app;

		/**
		 * @var int
		 * @since 1.0.0
		 */
		public $formId;

		/**
		 * @var array
		 * @since 1.0.0
		 */
		public $formColumns;

		/**
		 * @var string
		 * @since 1.0.0
		 */
		public $formName;


		/**
		 * @var boolean
		 * @since 1.0.0
		 */
		public $isMultiLang;

		/**
		 * Is this view an Empty State
		 *
		 * @var   boolean
		 * @since 1.0.0
		 */
		private $isEmptyState = false;

		/**
		 * Method to display the view.
		 *
		 * @param   string  $tpl  A template file to load. [optional]
		 *
		 * @return  void
		 *
		 * @throws  Exception
		 * @since   1.0.0
		 */
		public function display($tpl = null): void
		{
			/** @var SubmissionsModel $model */
			$model               = $this->getModel();
			$this->items         = $model->getItems();
			$this->pagination    = $model->getPagination();
			$this->filterForm    = $model->getFilterForm();
			$this->activeFilters = $model->getActiveFilters();
			$this->state         = $model->getState();
			$this->isEmptyState  = $this->get('IsEmptyState');
			$this->formId        = (int) $this->state->get('filter.form_id', 0);
			$this->app           = Factory::getApplication();
			$errors              = $this->getErrors();

			if (count($errors))
			{
				throw new GenericDataException(implode("\n", $errors), 500);
			}

			if (!count($this->items) && $this->isEmptyState)
			{
				$this->setLayout('emptystate');
			}
			$this->isMultiLang = Multilanguage::isEnabled();
			$this->_getFormColumns();


			// We don't need toolbar in the modal window.
			if ($this->getLayout() !== 'modal')
			{
				FormeaHelper::addSubmenu('formeas');
				$this->addToolbar();
				$this->sidebar = Sidebar::render();
			}


			parent::display($tpl);
		}

		/**
		 * Add the page title and toolbar.
		 *
		 * @return  void
		 *
		 * @throws  Exception
		 * @since   1.0.0
		 */
		protected function addToolbar()
		{
			$canDo = ContentHelper::getActions('com_formea',
				'category',
				$this->state->get('filter.category_id'));
			$user  = Factory::getApplication()->getIdentity();

			// Get the toolbar object instance
			$toolbar = Toolbar::getInstance('toolbar');

			ToolbarHelper::title(Text::_('COM_FORMEA_SUBMISSIONS_MANAGER'),
				'address formea');
			$this->getDocument()->setTitle(Text::_('COM_FORMEA_SUBMISSIONS_MANAGER') . ' - ' . Text::_('COM_FORMEA'));


			if ($canDo->get('core.delete'))
			{
				$toolbar->delete('submissions.delete')
					->text('JTOOLBAR_DELETE')
					->message('JGLOBAL_CONFIRM_DELETE')
					->listCheck(true);
			}

			if ($user->authorise('core.admin', 'com_formea')
				|| $user->authorise('core.options',
					'com_formea'))
			{
				$toolbar->preferences('com_formea');
			}

			ToolbarHelper::divider();
			ToolbarHelper::help('', false, FormeaHelper::getHelpLink('Submissions'));

			HTMLHelper::_('sidebar.setAction', 'index.php?option=com_formea');
		}


		/**
		 * Returns an array of fields the table can be sorted by
		 *
		 * @return  array  Array containing the field name to sort by as the key and display text as value
		 *
		 * @since   1.0.0
		 */
		protected function getSortFields()
		{
			return [
				'a.ordering'     => Text::_('JGRID_HEADING_ORDERING'),
				'a.published'    => Text::_('JSTATUS'),
				'a.name'         => Text::_('JGLOBAL_TITLE'),
				'category_title' => Text::_('JCATEGORY'),
				'a.access'       => Text::_('JGRID_HEADING_ACCESS'),
				'a.language'     => Text::_('JGRID_HEADING_LANGUAGE'),
				'a.id'           => Text::_('JGRID_HEADING_ID'),
			];
		}

		protected function _getFormColumns()
		{
			$this->formColumns = [];
			$this->formName    = '';
			if ($this->formId > 0)
			{
				$container = Factory::getContainer();
				/** @var DatabaseDriver $db */
				$db    = $container->get('DatabaseDriver');
				$query = $db->getQuery(true);
				$query->select(['admin_column', 'title']);
				$query->from($db->quoteName('#__formea_forms'));
				$query->where($db->quoteName('id') . ' = ' . $db->quote($this->formId));
				$db->setQuery($query);
				$formObject = $db->loadObject();
				if (!empty($formObject))
				{
					if (empty($formObject->admin_column))
					{
						$formObject->admin_column = '[]';
					}
					$admin_column_result = json_decode($formObject->admin_column);
					$this->formName      = $formObject->title;
					$elements            = [];
					foreach ($admin_column_result as $element_id => $enabled)
					{
						if ((int) $enabled > 0)
						{
							$elements[] = (int) $element_id;
						}
					}
					if (!empty($elements))
					{
						$elementLists = [];
						$query->clear();
						$query->select(['title', 'id']);
						$query->from($db->quoteName('#__formea_elements'));
						$query->where($db->quoteName('id') . ' IN (' . implode(',', $elements) . ')');
						$query->order('id ASC');
						$db->setQuery($query);
						$elementTitles = $db->loadObjectList();
						if (!empty($elementTitles))
						{
							$totalElements = count($elements);
							$totalTitles   = count($elementTitles);
							for ($i = 0; $i < $totalElements; $i++)
							{
								for ($j = 0; $j < $totalTitles; $j++)
								{
									if ((int) $elementTitles[$j]->id === (int) $elements[$i])
									{
										$elementLists[] = $elementTitles[$j];
										break;
									}
								}
							}
						}
						if (!empty($elementLists))
						{
							$this->formColumns = $elementLists;
						}
					}
				}
			}

			return $this->formColumns;
		}


	}
