<?php
    /**
     * @package     Joomla.Site
     * @subpackage  com_formea
     *
     * @copyright   Copyright (C) 2010-2024 Feseur Sdn Bhd. All rights reserved.
     * @license     GNU General Public License version 2 or later; see LICENSE.txt
     * @version     1.2.2
     * @since       1.2.0
     */

    namespace Joomla\Component\Formea\Site\Libraries;

    use Joomla\CMS\Date\Date;
    use Joomla\CMS\Factory;
    use Joomla\CMS\Helper\MediaHelper;
    use Joomla\CMS\HTML\HTMLHelper;
    use Joomla\CMS\Language\Text;
    use Joomla\CMS\Uri\Uri;
    use Joomla\CMS\User\User;
    use Joomla\Filesystem\Path;

    class FormeaAttachment
    {
        /**
         * The current user object.
         *
         * @var    User
         * @since  1.2.0
         */
        private $currentUser;


        public function __construct()
        {
        }

        /**
         * @param $paths
         *
         * @return FormeaAttachmentFile[]
         *
         * @since 1.2.0
         */
        public function getRelativeFiles($paths = [])
        {
            $totalPaths = count($paths);
            $processedFiles = [];
            for ($i = 0; $i < $totalPaths; $i++) {
                $cleanedPath = "";
                $proceed = false;
                $conf = [
                    'type' => 'file',
                    'name' => '',
                    'path' => $paths[$i],
                    'cleaned_path' => '',
                    'extension' => '',
                    'size' => 0,
                    'mime_type' => '',
                    'width' => 0,
                    'height' => 0,
                    'create_date' => '',
                    'create_date_formatted' => '',
                    'modified_date' => '',
                    'modified_date_formatted' => '',
                    'url' => '',
                    'thumb_path' => '',
                    'adapter' => '',
                ];
                if (strpos($paths[$i], 'local-images:/') === 0) {
                    //TODO change the path according to media manager parameter
                    //echo "The string starts with 'local-images:/'.";
                    $cleanedPath = str_replace('local-images:/', '', $paths[$i]);
                    $cleanedPath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'images' . DIRECTORY_SEPARATOR . $cleanedPath;
                    $conf['adapter'] = 'local-images';
                } elseif (strpos($paths[$i], '[') === 0) {
                    //placeholder
                    $conf['adapter'] = 'placeholder';
                    $conf['name'] = $paths[$i];
                    $processedFiles[] = new FormeaAttachmentFile($conf);
                } else {
                    //relative path
                    $conf['adapter'] = 'relative';
                    $cleanedPath = ltrim($paths[$i], '/');
                    $cleanedPath = JPATH_ROOT . DIRECTORY_SEPARATOR . $cleanedPath;
                }

                if ($conf['adapter'] !== 'placeholder') {
                    if (is_file(Path::clean($cleanedPath))) {
                        $fileInfo = pathinfo($cleanedPath);
                        $conf['cleaned_path'] = $cleanedPath;
                        $conf['name'] = $fileInfo['basename'];
                        if (isset($fileInfo['extension'])) {
                            $conf['extension'] = $fileInfo['extension'];
                        }


                        $createDate = $this->getDate(filectime($cleanedPath));
                        $modifiedDate = $this->getDate(filemtime($cleanedPath));
                        $isImage = MediaHelper::isImage($conf['name']);
                        $conf['size'] = filesize($cleanedPath);
                        $conf['mime_type'] = (string)MediaHelper::getMimeType($cleanedPath, $isImage);
                        $conf['create_date'] = $createDate->format('c', true);
                        $conf['create_date_formatted'] = HTMLHelper::_('date', $createDate, Text::_('DATE_FORMAT_LC5'));
                        $conf['modified_date'] = $modifiedDate->format('c', true);
                        $conf['modified_date_formatted'] = HTMLHelper::_('date', $modifiedDate, Text::_('DATE_FORMAT_LC5'));
                        $conf['url'] = rtrim(Uri::root(), '/') . str_replace(JPATH_ROOT, '', $cleanedPath);
                        if ($isImage) {
                            $conf['thumb_path'] = $conf['url'];
                        }
                        $processedFiles[] = new FormeaAttachmentFile($conf);
                    }

                }


            }

            return $processedFiles;
        }

        /**
         * Returns a Date with the correct Joomla timezone for the given date.
         *
         * @param string $date The date to create a Date from
         *
         * @return  Date
         *
         * @since   4.0.0
         */
        private function getDate($date = null): Date
        {
            $dateObj = Factory::getDate($date);

            $timezone = Factory::getApplication()->get('offset');
            $user = $this->getCurrentUser();

            if ($user->id) {
                $userTimezone = $user->getParam('timezone');

                if (!empty($userTimezone)) {
                    $timezone = $userTimezone;
                }
            }

            if ($timezone) {
                $dateObj->setTimezone(new \DateTimeZone($timezone));
            }

            return $dateObj;
        }

        private function getCurrentUser()
        {
            if (!$this->currentUser) {
                $this->currentUser = Factory::getApplication()->getIdentity();
            }

            return $this->currentUser;
        }

        public function getIconSets()
        {
            return [
                'pdf' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M64 464l48 0 0 48-48 0c-35.3 0-64-28.7-64-64L0 64C0 28.7 28.7 0 64 0L229.5 0c17 0 33.3 6.7 45.3 18.7l90.5 90.5c12 12 18.7 28.3 18.7 45.3L384 304l-48 0 0-144-80 0c-17.7 0-32-14.3-32-32l0-80L64 48c-8.8 0-16 7.2-16 16l0 384c0 8.8 7.2 16 16 16zM176 352l32 0c30.9 0 56 25.1 56 56s-25.1 56-56 56l-16 0 0 32c0 8.8-7.2 16-16 16s-16-7.2-16-16l0-48 0-80c0-8.8 7.2-16 16-16zm32 80c13.3 0 24-10.7 24-24s-10.7-24-24-24l-16 0 0 48 16 0zm96-80l32 0c26.5 0 48 21.5 48 48l0 64c0 26.5-21.5 48-48 48l-32 0c-8.8 0-16-7.2-16-16l0-128c0-8.8 7.2-16 16-16zm32 128c8.8 0 16-7.2 16-16l0-64c0-8.8-7.2-16-16-16l-16 0 0 96 16 0zm80-112c0-8.8 7.2-16 16-16l48 0c8.8 0 16 7.2 16 16s-7.2 16-16 16l-32 0 0 32 32 0c8.8 0 16 7.2 16 16s-7.2 16-16 16l-32 0 0 48c0 8.8-7.2 16-16 16s-16-7.2-16-16l0-64 0-64z"/></svg>',
                    'svgSafe' => '',
                ],
                'file' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><path d="M320 464c8.8 0 16-7.2 16-16l0-288-80 0c-17.7 0-32-14.3-32-32l0-80L64 48c-8.8 0-16 7.2-16 16l0 384c0 8.8 7.2 16 16 16l256 0zM0 64C0 28.7 28.7 0 64 0L229.5 0c17 0 33.3 6.7 45.3 18.7l90.5 90.5c12 12 18.7 28.3 18.7 45.3L384 448c0 35.3-28.7 64-64 64L64 512c-35.3 0-64-28.7-64-64L0 64z"/></svg>',
                    'svgSafe' => '',
                ],
                'compressed' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 384 512"><path d="M64 464c-8.8 0-16-7.2-16-16L48 64c0-8.8 7.2-16 16-16l48 0c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l48 0 0 80c0 17.7 14.3 32 32 32l80 0 0 288c0 8.8-7.2 16-16 16L64 464zM64 0C28.7 0 0 28.7 0 64L0 448c0 35.3 28.7 64 64 64l256 0c35.3 0 64-28.7 64-64l0-293.5c0-17-6.7-33.3-18.7-45.3L274.7 18.7C262.7 6.7 246.5 0 229.5 0L64 0zm48 112c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16s-7.2-16-16-16l-32 0c-8.8 0-16 7.2-16 16zm0 64c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16s-7.2-16-16-16l-32 0c-8.8 0-16 7.2-16 16zm-6.3 71.8L82.1 335.9c-1.4 5.4-2.1 10.9-2.1 16.4c0 35.2 28.8 63.7 64 63.7s64-28.5 64-63.7c0-5.5-.7-11.1-2.1-16.4l-23.5-88.2c-3.7-14-16.4-23.8-30.9-23.8l-14.8 0c-14.5 0-27.2 9.7-30.9 23.8zM128 336l32 0c8.8 0 16 7.2 16 16s-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16s7.2-16 16-16z"/></svg>',
                    'svgSafe' => '',
                ],
                'audio' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -960 960 960"><path d="M400-120q-66 0-113-47t-47-113q0-66 47-113t113-47q23 0 42.5 5.5T480-418v-422h240v160H560v400q0 66-47 113t-113 47Z"/></svg>',
                    'svgSafe' => '',
                ],
                'video' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -960 960 960"><path d="m160-800 80 160h120l-80-160h80l80 160h120l-80-160h80l80 160h120l-80-160h120q33 0 56.5 23.5T880-720v480q0 33-23.5 56.5T800-160H160q-33 0-56.5-23.5T80-240v-480q0-33 23.5-56.5T160-800Zm0 240v320h640v-320H160Zm0 0v320-320Z"/></svg>',
                    'svgSafe' => '',
                ],
                'attach' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -960 960 960"><path d="M720-330q0 104-73 177T470-80q-104 0-177-73t-73-177v-370q0-75 52.5-127.5T400-880q75 0 127.5 52.5T580-700v350q0 46-32 78t-78 32q-46 0-78-32t-32-78v-370h80v370q0 13 8.5 21.5T470-320q13 0 21.5-8.5T500-350v-350q-1-42-29.5-71T400-800q-42 0-71 29t-29 71v370q-1 71 49 120.5T470-160q70 0 119-49.5T640-330v-390h80v390Z"/></svg>',
                    //'svg'     => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><path d="M364.2 83.8c-24.4-24.4-64-24.4-88.4 0l-184 184c-42.1 42.1-42.1 110.3 0 152.4s110.3 42.1 152.4 0l152-152c10.9-10.9 28.7-10.9 39.6 0s10.9 28.7 0 39.6l-152 152c-64 64-167.6 64-231.6 0s-64-167.6 0-231.6l184-184c46.3-46.3 121.3-46.3 167.6 0s46.3 121.3 0 167.6l-176 176c-28.6 28.6-75 28.6-103.6 0s-28.6-75 0-103.6l144-144c10.9-10.9 28.7-10.9 39.6 0s10.9 28.7 0 39.6l-144 144c-6.7 6.7-6.7 17.7 0 24.4s17.7 6.7 24.4 0l176-176c24.4-24.4 24.4-64 0-88.4z"/></svg>',
                    'svgSafe' => '',
                ],
                'image' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M448 80c8.8 0 16 7.2 16 16l0 319.8-5-6.5-136-176c-4.5-5.9-11.6-9.3-19-9.3s-14.4 3.4-19 9.3L202 340.7l-30.5-42.7C167 291.7 159.8 288 152 288s-15 3.7-19.5 10.1l-80 112L48 416.3l0-.3L48 96c0-8.8 7.2-16 16-16l384 0zM64 32C28.7 32 0 60.7 0 96L0 416c0 35.3 28.7 64 64 64l384 0c35.3 0 64-28.7 64-64l0-320c0-35.3-28.7-64-64-64L64 32zm80 192a48 48 0 1 0 0-96 48 48 0 1 0 0 96z"/></svg>',
                    'svgSafe' => '',
                ],
                'folder' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M0 96C0 60.7 28.7 32 64 32l132.1 0c19.1 0 37.4 7.6 50.9 21.1L289.9 96 448 96c35.3 0 64 28.7 64 64l0 256c0 35.3-28.7 64-64 64L64 480c-35.3 0-64-28.7-64-64L0 96zM64 80c-8.8 0-16 7.2-16 16l0 320c0 8.8 7.2 16 16 16l384 0c8.8 0 16-7.2 16-16l0-256c0-8.8-7.2-16-16-16l-161.4 0c-10.6 0-20.8-4.2-28.3-11.7L213.1 87c-4.5-4.5-10.6-7-17-7L64 80z"/></svg>',
                    'svgSafe' => '',
                ],
                'x' => [
                    'svg' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 -960 960 960"><path d="m256-200-56-56 224-224-224-224 56-56 224 224 224-224 56 56-224 224 224 224-56 56-224-224-224 224Z"/></svg>',
                    'svgSafe' => '',
                ]
            ];
        }

    }
