<?php
  /**
   * @package     Joomla.Library
   * @subpackage  Feseur Library
   *
   * @copyright   Copyright (C) 2010 - 2022 Feseur Sdn Bhd. All rights reserved.
   * @license     GNU General Public License version 2 or later; see LICENSE.txt
   * @version     1.0.6
   */

  namespace Feseur\Library;

  defined('_JEXEC') or die;

  use DateInterval;
  use DatePeriod;
  use DateTime;
  use DOMDocument;
  use DOMXPath;
  use Exception;
  use Joomla\Archive\Archive;
  use Joomla\CMS\Application\ApplicationHelper;
  use Joomla\CMS\Factory;
  use Joomla\Filesystem\File;
  use Joomla\Filesystem\Folder;
  use Joomla\Filesystem\Path;
  use Joomla\CMS\HTML\HTMLHelper;
  use Joomla\CMS\Language\Text;
  use Joomla\Database\DatabaseDriver;

  class FsrHelper
  {

    /**
     * Get all months between 2 dates
     *
     * @param   string  $startDate
     * @param   string  $endDate
     * @param   string  $formatVal  PHP Date Format eg:Y-m-d.
     * @param   string  $formatKey  PHP Date Format eg:Y-m-d.
     *
     * @return array
     *
     * @throws Exception
     * @since 1.0.0
     */
    public static function getMonthsInRange($startDate, $endDate, $formatVal = 'Y-m', $formatKey = null)
    {
      $start    = (new DateTime($startDate))->modify('first day of this month');
      $end      = (new DateTime($endDate))->modify('first day of next month');
      $interval = DateInterval::createFromDateString('1 month');
      $period   = new DatePeriod($start, $interval, $end);
      $ret      = [];
      if (empty($formatKey))
      {
        foreach ($period as $dt)
        {
          $ret[] = $dt->format($formatVal);
        }
      }
      else
      {
        foreach ($period as $dt)
        {
          $ret[$dt->format($formatKey)] = $dt->format($formatVal);
        }
      }

      return $ret;

    }

    /**
     * Get common mime type for zip file
     * @return string[]
     *
     * @since 1.0.4
     */
    public static function commonZipMimeType()
    {
      $zipMimeType = [
        'application/x-zip',
        'application/x-zip-compressed',
        'application/octet-stream',
        'application/x-compress',
        'application/x-compressed',
        'multipart/x-zip',
        'application/zip',
      ];

      return $zipMimeType;
    }

    /**
     * Upload component custom plugins
     *
     * @param   array   $plugin
     * @param   string  $textPrefix
     * @param   string  $pluginType
     * @param   string  $destinationPath
     * @param   string  $groupType
     *
     * @return FsrResponse
     *
     * @since 1.0.0
     */
    public static function uploadFsrComPlugin($plugin, $textPrefix, $pluginType, $destinationPath, $groupType = null, $component_name = null)
    {
      $retObject = new FsrResponse();
      try
      {
        if (!isset($plugin['name']))
        {
          throw new Exception(Text::_($textPrefix . '_FILE_NOT_FOUND'));
        }
        if (!isset($plugin['type']) || !in_array($plugin['type'], self::commonZipMimeType()))
        {
          throw new Exception(Text::_($textPrefix . '_INVALID_FILE_TYPE') . " FLIB");
        }
        if ($plugin['error'] || $plugin['size'] < 1)
        {
          throw new Exception(Text::_($textPrefix . '_UPLOAD_PLUGIN_ERROR'));
        }
        $app     = Factory::getApplication();
        $tmpPath = $app->getConfig()->get('tmp_path');
        if (!is_dir(Path::clean($tmpPath)))
        {

          $tmpPath = JPATH_ROOT . DIRECTORY_SEPARATOR . 'tmp';
        }
        $destinationDir = $tmpPath . DIRECTORY_SEPARATOR . $plugin['name'];

        $uploaded = File::upload($plugin['tmp_name'], $destinationDir, false, true);

        if (!$uploaded)
        {
          throw new Exception(Text::_($textPrefix . '_UPLOAD_PLUGIN_FAILED'));
        }

        // Temporary folder to extract the archive into
        $tmpDir     = uniqid('install_');
        $extractDir = Path::clean(dirname($destinationDir) . DIRECTORY_SEPARATOR . $tmpDir);

        $archive = new Archive(['tmp_path' => $tmpPath]);
        $result  = $archive->extract($destinationDir, $extractDir);
        if (!$result)
        {
          throw new Exception(Text::_($textPrefix . '_EXTRACT_PLUGIN_ERROR'));
        }

        $dirList = array_merge(Folder::files($extractDir, ''), Folder::folders($extractDir, ''));

        if (isset($dirList) && count($dirList) == 1)
        {
          if (is_dir(Path::clean($extractDir . DIRECTORY_SEPARATOR . $dirList[0])))
          {
            $extractDir = Path::clean($extractDir . DIRECTORY_SEPARATOR . $dirList[0]);
          }
        }

        //Now, search for xml file
        $xmlFiles = Folder::files($extractDir, '.xml$', 1, true);

        if (empty($xmlFiles))
        {
          throw new Exception(Text::_($textPrefix . '_COULD_NOT_FIND_XML_FILE'));
        }

        $file = $xmlFiles[0];
        $root = simplexml_load_file($file);
        if (!$root)
        {
          throw new Exception(Text::_($textPrefix . '_INVALID_FILE'));
        }

        if ($root->getName() !== 'fsrExtension')
        {
          throw new Exception(Text::_($textPrefix . '_INVALID_XML_FILE'));
        }
        $_xmlArrayData = self::xmlToArray($root);
        $xmlArrayData  = $_xmlArrayData['fsrExtension'];
        $_pluginType   = (string) $root->attributes()->type;
        $_groupType    = (string) $root->attributes()->group;

        $xmlArrayData['type']  = $_pluginType;
        $xmlArrayData['group'] = $_groupType;
        if (!empty($groupType))
        {
          if ($_pluginType != $pluginType || $_groupType != $groupType)
          {
            throw new Exception(Text::_($textPrefix . '_INVALID_' . strtoupper($groupType) . '_PLUGIN'));
          }
        }
        else
        {
          if ($_pluginType != $pluginType)
          {
            throw new Exception(Text::_($textPrefix . '_INVALID_COMPONENT_PLUGIN'));
          }
        }
        $name     = (string) $root->name;
        $mediaDir = '';
        if (!empty($component_name))
        {
          $mediaDir = JPATH_ROOT . DIRECTORY_SEPARATOR . 'media' . DIRECTORY_SEPARATOR . $component_name;
        }
        if (empty($groupType))
        {
          $pluginDir = rtrim($destinationPath, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $_groupType . DIRECTORY_SEPARATOR . $name;
        }
        else
        {
          $pluginDir = rtrim($destinationPath, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $groupType . DIRECTORY_SEPARATOR . $name;
        }
        if (!is_dir(Path::clean($pluginDir)))
        {
          Folder::create($pluginDir);
        }
        //move the xml plugin file
        File::move($file, $pluginDir . DIRECTORY_SEPARATOR . basename($file));
        $files = $root->files->children();

        if (isset($files))
        {
          for ($i = 0, $n = count($files); $i < $n; $i++)
          {
            $file = $files[$i];
            if ($file->getName() == 'filename')
            {
              $fileName = $file;
              File::copy($extractDir . DIRECTORY_SEPARATOR . $fileName, $pluginDir . DIRECTORY_SEPARATOR . $fileName);
            }
            elseif ($file->getName() == 'folder')
            {
              $folderName = $file;
              if (is_dir(Path::clean($extractDir . DIRECTORY_SEPARATOR . $folderName)))
              {
                if ((string) $folderName === 'media' && !empty($component_name))
                {
                  $mediaFiles   = Folder::files($extractDir . DIRECTORY_SEPARATOR . $folderName);
                  $mediaFolders = Folder::folders($extractDir . DIRECTORY_SEPARATOR . $folderName);
                  if (!empty($mediaFiles))
                  {
                    $totalMediaFiles = count($mediaFiles);
                    for ($j = 0; $j < $totalMediaFiles; $j++)
                    {
                      File::copy($extractDir . DIRECTORY_SEPARATOR . $folderName . DIRECTORY_SEPARATOR . $mediaFiles[$j], $mediaDir . DIRECTORY_SEPARATOR . $mediaFiles[$j]);
                    }
                  }
                  if (!empty($mediaFolders))
                  {
                    $totalMediaFolders = count($mediaFolders);
                    for ($j = 0; $j < $totalMediaFolders; $j++)
                    {
                      Folder::copy($extractDir . DIRECTORY_SEPARATOR . $folderName . DIRECTORY_SEPARATOR . $mediaFolders[$j], $mediaDir . DIRECTORY_SEPARATOR . $mediaFolders[$j], '', true, true);
                    }
                  }
                }
                else
                {
                  if (is_dir(Path::clean($pluginDir . DIRECTORY_SEPARATOR . $folderName)))
                  {
                    Folder::delete($pluginDir . DIRECTORY_SEPARATOR . $folderName);
                  }
                  Folder::move($extractDir . DIRECTORY_SEPARATOR . $folderName, $pluginDir . DIRECTORY_SEPARATOR . $folderName);
                }
              }
            }
          }
        }

        Folder::delete($extractDir);
        if (is_file(Path::clean($destinationDir)))
        {
          File::delete($destinationDir);
        }
        $retObject->setSuccess(true);
        $retObject->setResult($xmlArrayData);

      }
      catch (Exception $e)
      {
        $retObject->setSuccess(false);
        $retObject->setMsg($e->getMessage());
      }

      return $retObject;
    }

    /**
     * Convert XML elements into array
     *
     * @param $xml
     * @param $options
     *
     * @return array|string[]
     *
     * @since 1.0.0
     */
    public static function xmlToArray($xml, $options = array())
    {
      $defaults = array(
        'namespaceSeparator' => ':',//you may want this to be something other than a colon
        'attributePrefix'    => '@',   //to distinguish between attributes and nodes with the same name
        'alwaysArray'        => array(),   //array of xml tag names which should always become arrays
        'autoArray'          => true,        //only create arrays for tags which appear more than once
        'textContent'        => '$',       //key used for the text content of elements
        'autoText'           => true,         //skip textContent key if node has no attributes or child nodes
        'keySearch'          => false,       //optional search and replace on tag and attribute names
        'keyReplace'         => false       //replace values for above search values (as passed to str_replace())
      );
      $options  = array_merge($defaults, $options);

      $namespaces     = $xml->getDocNamespaces();
      $namespaces[''] = null; //add base (empty) namespace

      //get attributes from all namespaces
      $attributesArray = array();
      foreach ($namespaces as $prefix => $namespace)
      {
        foreach ($xml->attributes($namespace) as $attributeName => $attribute)
        {
          //replace characters in attribute name
          if ($options['keySearch']) $attributeName =
            str_replace($options['keySearch'], $options['keyReplace'], $attributeName);
          $attributeKey                   = $options['attributePrefix']
            . ($prefix ? $prefix . $options['namespaceSeparator'] : '')
            . $attributeName;
          $attributesArray[$attributeKey] = (string) $attribute;
        }
      }

      //get child nodes from all namespaces
      $tagsArray = array();
      foreach ($namespaces as $prefix => $namespace)
      {
        foreach ($xml->children($namespace) as $childXml)
        {
          //recurse into child nodes
          $childArray = self::xmlToArray($childXml, $options);
          foreach ($childArray as $childTagName => $childProperties)
          {
            //replace characters in tag name
            if ($options['keySearch']) $childTagName = str_replace($options['keySearch'], $options['keyReplace'], $childTagName);
            //add namespace prefix, if any
            if ($prefix) $childTagName = $prefix . $options['namespaceSeparator'] . $childTagName;

            if (!isset($tagsArray[$childTagName]))
            {
              //only entry with this key
              //test if tags of this type should always be arrays, no matter the element count
              $tagsArray[$childTagName] =
                in_array($childTagName, $options['alwaysArray']) || !$options['autoArray']
                  ? array($childProperties) : $childProperties;
            }
            elseif (
              is_array($tagsArray[$childTagName]) && array_keys($tagsArray[$childTagName])
              === range(0, count($tagsArray[$childTagName]) - 1)
            )
            {
              //key already exists and is integer indexed array
              $tagsArray[$childTagName][] = $childProperties;
            }
            else
            {
              //key exists so convert to integer indexed array with previous value in position 0
              $tagsArray[$childTagName] = array($tagsArray[$childTagName], $childProperties);
            }
          }
        }
      }

      //get text content of node
      $textContentArray = array();
      $plainText        = trim((string) $xml);
      if ($plainText !== '') $textContentArray[$options['textContent']] = $plainText;

      //stick it all together
      $propertiesArray = !$options['autoText'] || $attributesArray || $tagsArray || ($plainText === '')
        ? array_merge($attributesArray, $tagsArray, $textContentArray) : $plainText;

      //return node as array
      return array(
        $xml->getName() => $propertiesArray
      );
    }

    /**
     * Create a unique alias/slug from a string, usually for a title
     *
     * @param   string  $alias
     * @param   string  $tableName
     * @param   int     $id
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function createAlias($alias, $tableName, $id = 0)
    {
      $alias     = ApplicationHelper::stringURLSafe($alias);
      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select('COUNT(*)');
      $query->from($db->quoteName('#__' . $tableName));
      $query->where($db->quoteName('alias') . ' = ' . $db->quote($alias));
      $query->where($db->quoteName('id') . ' != ' . $db->quote($id));
      $db->setQuery($query);
      $found = (int) $db->loadResult();
      if ($found > 0)
      {
        //$d     = new FsrDate();
        //$alias .= $d->toSql();
        $alias .= '-'.($found + 1);
        $alias = ApplicationHelper::stringURLSafe($alias);
      }

      return $alias;
    }

    /**
     * Get the language string in associative key => value pair array
     *
     * @param   boolean   $isAdmin                 administrator files or client files (Front end)
     * @param   string[]  $additionalLanguagePath  additional language file to load
     * @param   string    $langTag                 language code en-GB
     * @param   boolean   $raw                     parse the language key with Joomla Text
     * @param   boolean   $excludeJoomla           whether to include joomla ini files
     *
     * @return array
     *
     * @throws Exception
     * @since 1.0.0
     */
    public static function getCommonLanguage($isAdmin = false, $additionalLanguagePath = [], $langTag = null, $raw = false, $excludeJoomla = false)
    {
      $defaultLangTag = 'en-GB';
      if (empty($langTag))
      {
        $langTag = Factory::getApplication()->getLanguage()->getTag();
      }
      $fileNames = [];
      if (!$excludeJoomla)
      {
        $joomla_ini = JPATH_ROOT . '/language/' . $langTag . '/joomla.ini';
        if ($isAdmin)
        {
          $joomla_ini = JPATH_ROOT . '/administrator/language/' . $langTag . '/joomla.ini';
        }
        $fileNames[] = $joomla_ini;
      }
      if (!empty($additionalLanguagePath))
      {
        $fileNames = array_merge($additionalLanguagePath, $fileNames);
      }
      $layoutTextString = [];
      $totalFilenames   = count($fileNames);
      for ($j = 0; $j < $totalFilenames; $j++)
      {
        $fileLookup = $fileNames[$j];
        if (!is_file($fileNames[$j]))
        {
          //change to default langTag
          $fileNames[$j] = str_replace($langTag, $defaultLangTag, $fileNames[$j]);
          if (is_file($fileNames[$j]))
          {
            $fileLookup = $fileNames[$j];
          }
          else
          {
            $fileLookup = null;
          }
        }
        if (!empty($fileLookup))
        {
          $langFile = parse_ini_file($fileNames[$j]);
          if ($raw)
          {
            $layoutTextString = array_merge($layoutTextString, $langFile);
          }
          else
          {
            $keys      = array_keys($langFile);
            $totalKeys = count($keys);
            for ($i = 0; $i < $totalKeys; $i++)
            {
              $layoutTextString[$keys[$i]] = Text::_($keys[$i]);
            }
          }

        }
      }

      return $layoutTextString;
    }

    public static function generateRandomUniqueString($length = 10, $existingStrings = [])
    {
      $characters   = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
      $randomString = '';

      do
      {
        $randomString = '';
        for ($i = 0; $i < $length; $i++)
        {
          $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
      } while (in_array($randomString, $existingStrings));

      return $randomString;
    }

    public static function findItemByKey($array, $keyToFind)
    {
      foreach ($array as $key => $value)
      {
        if ($key === $keyToFind)
        {
          return $value;
        }
        elseif (is_array($value))
        {
          $result = self::findItemByKey($value, $keyToFind);
          if ($result !== null)
          {
            return $result;
          }
        }
      }

      return null; // Key not found in the array
    }

    public static function findItemByKeyAndValue($array, $keyToFind, $valueToFind)
    {
      foreach ($array as $key => $value)
      {
        if ($key === $keyToFind && $value === $valueToFind)
        {
          return $array;
        }
        elseif (is_array($value))
        {
          $result = self::findItemByKeyAndValue($value, $keyToFind, $valueToFind);
          if ($result !== null)
          {
            return $result;
          }
        }
      }

      return null; // Key and value not found in the array
    }

    public static function createJoomlaXmlField($array)
    {
      $ret = '';
      foreach ($array as $field)
      {
        $hasChild = false;
        $ret      .= '<field ';
        foreach ($field as $attr => $attrValue)
        {
          if (substr($attr, 0, 1) === "@")
          {
            //attribute
            $ret .= str_replace('@', '', $attr) . '="' . $attrValue . '" ';
          }
          if ($attr === 'option')
          {
            $ret      .= '>';
            $hasChild = true;
            foreach ($attrValue as $v)
            {
              $ret .= '<option value="' . $v['@value'] . '">' . $v['$'] . '</option>';
            }
          }
        }
        if ($hasChild)
        {
          $ret .= '</field>';
        }
        else
        {
          $ret .= '/>';
        }
      }

      return $ret;
    }

    /**
     * Method to generate random string
     *
     * @param $length
     *
     * @return string
     *
     * @since 1.0.0
     */
    public static function generateRandomString($length = 10)
    {
      $characters   = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
      $randomString = '';

      for ($i = 0; $i < $length; $i++)
      {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
      }

      return $randomString;
    }

    public static function convertBytesToHumanReadable($bytes)
    {
      $sizeUnits = ['B', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB'];
      $unitIndex = 0;

      while ($bytes >= 1024 && $unitIndex < count($sizeUnits) - 1)
      {
        $bytes /= 1024;
        $unitIndex++;
      }

      $formattedSize = number_format($bytes, 1);
      $unit          = $sizeUnits[$unitIndex];

      return $formattedSize . ' ' . $unit;
    }

    /**
     * @param string $size Eg. '8M' or '32K'
     *
     * @return float|int
     *
     * @since 1.0.6
     */
    public static function convertToBytes($size) {
      // Get the unit of the size (e.g., 'M', 'G', 'K')
      $unit = strtoupper(substr($size, -1));
      $bytes = (int) $size;

      switch ($unit) {
        case 'G':
          $bytes *= 1024 * 1024 * 1024; // Gigabytes to bytes
          break;
        case 'M':
          $bytes *= 1024 * 1024; // Megabytes to bytes
          break;
        case 'K':
          $bytes *= 1024; // Kilobytes to bytes
          break;
      }

      return $bytes;
    }

    public static function getImageSrc($image)
    {
      $ret = '';
      if (!empty($image))
      {
        $image = HTMLHelper::_('cleanImageURL', $image);
        if (isset($image->url))
        {
          $ret = $image->url;
        }
      }

      return $ret;
    }

    /**
     * Method to list all <img> source from a html in an array.
     * @param string $html
     *
     * @return array
     *
     * @since 1.0.6
     */
    public static function getAllImageSrc($html) {
      $dom = new DOMDocument();
      // Suppress errors due to malformed HTML by using @
      @$dom->loadHTML($html);

      // Create an XPath object to query the DOM
      $xpath = new DOMXPath($dom);

      // Query for all <img> elements
      $images = $xpath->query('//img');

      // Initialize an array to hold the src values
      $srcArray = [];

      // Loop through each <img> element and extract the src attribute
      foreach ($images as $image) {
        $src = $image->getAttribute('src');
        $srcArray[] = $src;
      }

      return $srcArray;
    }

    public static function copyFilesFromBackup($sourceDir,$sourceDirArray = [],$fileArray = []){
      $totalImgArray = count($fileArray);
      for($k = 0; $k < $totalImgArray; $k++) {

        $fileInfo =pathinfo($fileArray[$k]);
        $originalFileDir = null;
        $fileNameWithExt = null;
        if(isset($fileInfo['basename'])){
          $fileNameWithExt = $fileInfo['basename'];
        }
        if(isset($fileInfo['dirname'])){
          $originalFileDir = $fileInfo['dirname'];
        }
        if(!empty($fileNameWithExt) && !empty($originalFileDir)){
          //check if file exists in source array
          if (in_array($fileNameWithExt, $sourceDirArray)){
              $sourcePath = null;
              $destPath = null;
            if (strpos($fileArray[$k], 'local-images:/') === 0) {
              //destination folder should be images, no need to change anything
              $sourcePath = $sourceDir.$fileNameWithExt;
              $destPath = str_replace('local-images:', JPATH_ROOT.'/images', $originalFileDir);
            }elseif (strpos($fileArray[$k], 'http://') === 0 || strpos($fileArray[$k], 'https://') === 0 || strpos($fileArray[$k], 'www.') === 0) {
              //no transfer, no need to change anything
            }elseif (strpos($fileArray[$k], '[') === 0){
              //placeholder, no need to do anything
            }else{
              //probably a relative path
              $sourcePath = $sourceDir.$fileNameWithExt;
              $destPath = JPATH_ROOT.'/'.$originalFileDir;
            }

            if(!empty($sourcePath) && !empty($destPath)){
              if(!is_dir(Path::clean($destPath))){
                Folder::create($destPath);
              }
              File::copy($sourcePath, $destPath.'/'.$fileNameWithExt);
            }

          }


        }
      }
      return true;
    }


    /**
     * Get the language string in associative key => value pair array
     *
     * @param   string[]  $additionalLanguagePath
     *
     * @return array
     *
     * @throws Exception
     * @since 1.0.0
     */
    public static function getCommonLanguageV2($lang, $loadFiles = [], $langTag = 'en-GB', $extraCoreFile = [])
    {
      $totalFiles   = count($loadFiles);
      $languageText = [];
      if (!empty($langTag))
      {
        $langTag = trim($langTag);
      }
      if (empty($langTag))
      {
        $langTag = 'en-GB';
      }
      $coreFiles = [
        'joomla.ini',
        'lib_joomla.ini'
      ];
      if (!empty($extraCoreFile))
      {
        $coreFiles = array_merge($coreFiles, $extraCoreFile);
      }
      $totalCoreFiles = count($coreFiles);
      for ($k = 0; $k < $totalCoreFiles; $k++)
      {
        //$lang->load($coreFiles[$k], $langTag, true);
        $langFileName = $coreFiles[$k];
        $langFile     = parse_ini_file(JPATH_ROOT . '/language/' . $langTag . '/' . $langFileName);
        $keys         = array_keys($langFile);
        $totalKeys    = count($keys);
        for ($i = 0; $i < $totalKeys; $i++)
        {
          $languageText[$keys[$i]] = Text::_($keys[$i]);
        }
      }


      for ($i = 0; $i < $totalFiles; $i++)
      {
        $lang->load($loadFiles[$i]->extension, $loadFiles[$i]->path, $langTag, true);
        $langFileName = $langTag . '.' . $loadFiles[$i]->extension . '.ini';
        $langFile     = parse_ini_file($loadFiles[$i]->path . '/language/' . $langTag . '/' . $langFileName);

        $keys      = array_keys($langFile);
        $totalKeys = count($keys);
        for ($j = 0; $j < $totalKeys; $j++)
        {
          $languageText[$keys[$j]] = Text::_($keys[$j]);
        }
      }

      return $languageText;

    }


    /**
     *
     * Function to read file
     *
     * @param   string   $filename
     * @param   boolean  $retbytes
     *
     * @return boolean|number
     * @since 1.0.0
     */
    public static function readfile_chunked($filename, $retbytes = true)
    {
      $chunksize = 1 * (1024 * 1024); // how many bytes per chunk
      $buffer    = '';
      $cnt       = 0;
      $handle    = fopen($filename, 'rb');

      if ($handle === false)
      {
        return false;
      }

      while (!feof($handle))
      {
        $buffer = fread($handle, $chunksize);
        echo $buffer;
        @ob_flush();
        flush();

        if ($retbytes)
        {
          $cnt += strlen($buffer);
        }
      }

      $status = fclose($handle);

      if ($retbytes && $status)
      {
        return $cnt; // return num. bytes delivered like readfile() does.
      }

      return $status;
    }


    /**
     *
     * Function to get mimetype of file
     *
     * @param   string  $ext
     *
     * @return string
     * @since 1.0.1
     */
    public static function getMimeType($ext)
    {
      $mime_extension_map = array('__MAXPERIOD__' => '1',
                                  '3ds'           => 'image/x-3ds',
                                  'BLEND'         => 'application/x-blender',
                                  'C'             => 'text/x-c++src',
                                  'CSSL'          => 'text/css',
                                  'NSV'           => 'video/x-nsv',
                                  'XM'            => 'audio/x-mod',
                                  'Z'             => 'application/x-compress',
                                  'a'             => 'application/x-archive',
                                  'abw'           => 'application/x-abiword',
                                  'abw.gz'        => 'application/x-abiword',
                                  'ac3'           => 'audio/ac3',
                                  'adb'           => 'text/x-adasrc',
                                  'ads'           => 'text/x-adasrc',
                                  'afm'           => 'application/x-font-afm',
                                  'ag'            => 'image/x-applix-graphics',
                                  'ai'            => 'application/illustrator',
                                  'aif'           => 'audio/x-aiff',
                                  'aifc'          => 'audio/x-aiff',
                                  'aiff'          => 'audio/x-aiff',
                                  'al'            => 'application/x-perl',
                                  'arj'           => 'application/x-arj',
                                  'as'            => 'application/x-applix-spreadsheet',
                                  'asc'           => 'text/plain',
                                  'asf'           => 'video/x-ms-asf',
                                  'asp'           => 'application/x-asp',
                                  'asx'           => 'video/x-ms-asf',
                                  'au'            => 'audio/basic',
                                  'avi'           => 'video/x-msvideo',
                                  'aw'            => 'application/x-applix-word',
                                  'bak'           => 'application/x-trash',
                                  'bcpio'         => 'application/x-bcpio',
                                  'bdf'           => 'application/x-font-bdf',
                                  'bib'           => 'text/x-bibtex',
                                  'bin'           => 'application/octet-stream',
                                  'blend'         => 'application/x-blender',
                                  'blender'       => 'application/x-blender',
                                  'bmp'           => 'image/bmp',
                                  'bz'            => 'application/x-bzip',
                                  'bz2'           => 'application/x-bzip',
                                  'c'             => 'text/x-csrc',
                                  'c++'           => 'text/x-c++src',
                                  'cc'            => 'text/x-c++src',
                                  'cdf'           => 'application/x-netcdf',
                                  'cdr'           => 'application/vnd.corel-draw',
                                  'cer'           => 'application/x-x509-ca-cert',
                                  'cert'          => 'application/x-x509-ca-cert',
                                  'cgi'           => 'application/x-cgi',
                                  'cgm'           => 'image/cgm',
                                  'chrt'          => 'application/x-kchart',
                                  'class'         => 'application/x-java',
                                  'cls'           => 'text/x-tex',
                                  'cpio'          => 'application/x-cpio',
                                  'cpio.gz'       => 'application/x-cpio-compressed',
                                  'cpp'           => 'text/x-c++src',
                                  'cpt'           => 'application/mac-compactpro',
                                  'crt'           => 'application/x-x509-ca-cert',
                                  'cs'            => 'text/x-csharp',
                                  'csh'           => 'application/x-shellscript',
                                  'css'           => 'text/css',
                                  'csv'           => 'text/x-comma-separated-values',
                                  'cur'           => 'image/x-win-bitmap',
                                  'cxx'           => 'text/x-c++src',
                                  'dat'           => 'video/mpeg',
                                  'dbf'           => 'application/x-dbase',
                                  'dc'            => 'application/x-dc-rom',
                                  'dcl'           => 'text/x-dcl',
                                  'dcm'           => 'image/x-dcm',
                                  'dcr'           => 'application/x-director',
                                  'deb'           => 'application/x-deb',
                                  'der'           => 'application/x-x509-ca-cert',
                                  'desktop'       => 'application/x-desktop',
                                  'dia'           => 'application/x-dia-diagram',
                                  'diff'          => 'text/x-patch',
                                  'dir'           => 'application/x-director',
                                  'djv'           => 'image/vnd.djvu',
                                  'djvu'          => 'image/vnd.djvu',
                                  'dll'           => 'application/octet-stream',
                                  'dms'           => 'application/octet-stream',
                                  'doc'           => 'application/msword',
                                  'dsl'           => 'text/x-dsl',
                                  'dtd'           => 'text/x-dtd',
                                  'dvi'           => 'application/x-dvi',
                                  'dwg'           => 'image/vnd.dwg',
                                  'dxf'           => 'image/vnd.dxf',
                                  'dxr'           => 'application/x-director',
                                  'egon'          => 'application/x-egon',
                                  'el'            => 'text/x-emacs-lisp',
                                  'eps'           => 'image/x-eps',
                                  'epsf'          => 'image/x-eps',
                                  'epsi'          => 'image/x-eps',
                                  'etheme'        => 'application/x-e-theme',
                                  'etx'           => 'text/x-setext',
                                  'exe'           => 'application/x-executable',
                                  'ez'            => 'application/andrew-inset',
                                  'f'             => 'text/x-fortran',
                                  'fig'           => 'image/x-xfig',
                                  'fits'          => 'image/x-fits',
                                  'flac'          => 'audio/x-flac',
                                  'flc'           => 'video/x-flic',
                                  'fli'           => 'video/x-flic',
                                  'flw'           => 'application/x-kivio',
                                  'fo'            => 'text/x-xslfo',
                                  'g3'            => 'image/fax-g3',
                                  'gb'            => 'application/x-gameboy-rom',
                                  'gcrd'          => 'text/x-vcard',
                                  'gen'           => 'application/x-genesis-rom',
                                  'gg'            => 'application/x-sms-rom',
                                  'gif'           => 'image/gif',
                                  'glade'         => 'application/x-glade',
                                  'gmo'           => 'application/x-gettext-translation',
                                  'gnc'           => 'application/x-gnucash',
                                  'gnucash'       => 'application/x-gnucash',
                                  'gnumeric'      => 'application/x-gnumeric',
                                  'gra'           => 'application/x-graphite',
                                  'gsf'           => 'application/x-font-type1',
                                  'gtar'          => 'application/x-gtar',
                                  'gz'            => 'application/x-gzip',
                                  'h'             => 'text/x-chdr',
                                  'h++'           => 'text/x-chdr',
                                  'hdf'           => 'application/x-hdf',
                                  'hh'            => 'text/x-c++hdr',
                                  'hp'            => 'text/x-chdr',
                                  'hpgl'          => 'application/vnd.hp-hpgl',
                                  'hqx'           => 'application/mac-binhex40',
                                  'hs'            => 'text/x-haskell',
                                  'htm'           => 'text/html',
                                  'html'          => 'text/html',
                                  'icb'           => 'image/x-icb',
                                  'ice'           => 'x-conference/x-cooltalk',
                                  'ico'           => 'image/x-ico',
                                  'ics'           => 'text/calendar',
                                  'idl'           => 'text/x-idl',
                                  'ief'           => 'image/ief',
                                  'ifb'           => 'text/calendar',
                                  'iff'           => 'image/x-iff',
                                  'iges'          => 'model/iges',
                                  'igs'           => 'model/iges',
                                  'ilbm'          => 'image/x-ilbm',
                                  'iso'           => 'application/x-cd-image',
                                  'it'            => 'audio/x-it',
                                  'jar'           => 'application/x-jar',
                                  'java'          => 'text/x-java',
                                  'jng'           => 'image/x-jng',
                                  'jp2'           => 'image/jpeg2000',
                                  'jpg'           => 'image/jpeg',
                                  'jpe'           => 'image/jpeg',
                                  'jpeg'          => 'image/jpeg',
                                  'jpr'           => 'application/x-jbuilder-project',
                                  'jpx'           => 'application/x-jbuilder-project',
                                  'js'            => 'application/x-javascript',
                                  'kar'           => 'audio/midi',
                                  'karbon'        => 'application/x-karbon',
                                  'kdelnk'        => 'application/x-desktop',
                                  'kfo'           => 'application/x-kformula',
                                  'kil'           => 'application/x-killustrator',
                                  'kon'           => 'application/x-kontour',
                                  'kpm'           => 'application/x-kpovmodeler',
                                  'kpr'           => 'application/x-kpresenter',
                                  'kpt'           => 'application/x-kpresenter',
                                  'kra'           => 'application/x-krita',
                                  'ksp'           => 'application/x-kspread',
                                  'kud'           => 'application/x-kugar',
                                  'kwd'           => 'application/x-kword',
                                  'kwt'           => 'application/x-kword',
                                  'la'            => 'application/x-shared-library-la',
                                  'latex'         => 'application/x-latex',
                                  'lha'           => 'application/x-lha',
                                  'lhs'           => 'text/x-literate-haskell',
                                  'lhz'           => 'application/x-lhz',
                                  'log'           => 'text/x-log',
                                  'ltx'           => 'text/x-tex',
                                  'lwo'           => 'image/x-lwo',
                                  'lwob'          => 'image/x-lwo',
                                  'lws'           => 'image/x-lws',
                                  'lyx'           => 'application/x-lyx',
                                  'lzh'           => 'application/x-lha',
                                  'lzo'           => 'application/x-lzop',
                                  'm'             => 'text/x-objcsrc',
                                  'm15'           => 'audio/x-mod',
                                  'm3u'           => 'audio/x-mpegurl',
                                  'man'           => 'application/x-troff-man',
                                  'md'            => 'application/x-genesis-rom',
                                  'me'            => 'text/x-troff-me',
                                  'mesh'          => 'model/mesh',
                                  'mgp'           => 'application/x-magicpoint',
                                  'mid'           => 'audio/midi',
                                  'midi'          => 'audio/midi',
                                  'mif'           => 'application/x-mif',
                                  'mkv'           => 'application/x-matroska',
                                  'mm'            => 'text/x-troff-mm',
                                  'mml'           => 'text/mathml',
                                  'mng'           => 'video/x-mng',
                                  'moc'           => 'text/x-moc',
                                  'mod'           => 'audio/x-mod',
                                  'moov'          => 'video/quicktime',
                                  'mov'           => 'video/quicktime',
                                  'movie'         => 'video/x-sgi-movie',
                                  'mp2'           => 'video/mpeg',
                                  'mp3'           => 'audio/x-mp3',
                                  'mpe'           => 'video/mpeg',
                                  'mpeg'          => 'video/mpeg',
                                  'mpg'           => 'video/mpeg',
                                  'mpga'          => 'audio/mpeg',
                                  'ms'            => 'text/x-troff-ms',
                                  'msh'           => 'model/mesh',
                                  'msod'          => 'image/x-msod',
                                  'msx'           => 'application/x-msx-rom',
                                  'mtm'           => 'audio/x-mod',
                                  'mxu'           => 'video/vnd.mpegurl',
                                  'n64'           => 'application/x-n64-rom',
                                  'nc'            => 'application/x-netcdf',
                                  'nes'           => 'application/x-nes-rom',
                                  'nsv'           => 'video/x-nsv',
                                  'o'             => 'application/x-object',
                                  'obj'           => 'application/x-tgif',
                                  'oda'           => 'application/oda',
                                  'odb'           => 'application/vnd.oasis.opendocument.database',
                                  'odc'           => 'application/vnd.oasis.opendocument.chart',
                                  'odf'           => 'application/vnd.oasis.opendocument.formula',
                                  'odg'           => 'application/vnd.oasis.opendocument.graphics',
                                  'odi'           => 'application/vnd.oasis.opendocument.image',
                                  'odm'           => 'application/vnd.oasis.opendocument.text-master',
                                  'odp'           => 'application/vnd.oasis.opendocument.presentation',
                                  'ods'           => 'application/vnd.oasis.opendocument.spreadsheet',
                                  'odt'           => 'application/vnd.oasis.opendocument.text',
                                  'ogg'           => 'application/ogg',
                                  'old'           => 'application/x-trash',
                                  'oleo'          => 'application/x-oleo',
                                  'otg'           => 'application/vnd.oasis.opendocument.graphics-template',
                                  'oth'           => 'application/vnd.oasis.opendocument.text-web',
                                  'otp'           => 'application/vnd.oasis.opendocument.presentation-template',
                                  'ots'           => 'application/vnd.oasis.opendocument.spreadsheet-template',
                                  'ott'           => 'application/vnd.oasis.opendocument.text-template',
                                  'p'             => 'text/x-pascal',
                                  'p12'           => 'application/x-pkcs12',
                                  'p7s'           => 'application/pkcs7-signature',
                                  'pas'           => 'text/x-pascal',
                                  'patch'         => 'text/x-patch',
                                  'pbm'           => 'image/x-portable-bitmap',
                                  'pcd'           => 'image/x-photo-cd',
                                  'pcf'           => 'application/x-font-pcf',
                                  'pcf.Z'         => 'application/x-font-type1',
                                  'pcl'           => 'application/vnd.hp-pcl',
                                  'pdb'           => 'application/vnd.palm',
                                  'pdf'           => 'application/pdf',
                                  'pem'           => 'application/x-x509-ca-cert',
                                  'perl'          => 'application/x-perl',
                                  'pfa'           => 'application/x-font-type1',
                                  'pfb'           => 'application/x-font-type1',
                                  'pfx'           => 'application/x-pkcs12',
                                  'pgm'           => 'image/x-portable-graymap',
                                  'pgn'           => 'application/x-chess-pgn',
                                  'pgp'           => 'application/pgp',
                                  'php'           => 'application/x-php',
                                  'php3'          => 'application/x-php',
                                  'php4'          => 'application/x-php',
                                  'pict'          => 'image/x-pict',
                                  'pict1'         => 'image/x-pict',
                                  'pict2'         => 'image/x-pict',
                                  'pl'            => 'application/x-perl',
                                  'pls'           => 'audio/x-scpls',
                                  'pm'            => 'application/x-perl',
                                  'png'           => 'image/png',
                                  'pnm'           => 'image/x-portable-anymap',
                                  'po'            => 'text/x-gettext-translation',
                                  'pot'           => 'application/vnd.ms-powerpoint',
                                  'ppm'           => 'image/x-portable-pixmap',
                                  'pps'           => 'application/vnd.ms-powerpoint',
                                  'ppt'           => 'application/vnd.ms-powerpoint',
                                  'ppz'           => 'application/vnd.ms-powerpoint',
                                  'ps'            => 'application/postscript',
                                  'ps.gz'         => 'application/x-gzpostscript',
                                  'psd'           => 'image/x-psd',
                                  'psf'           => 'application/x-font-linux-psf',
                                  'psid'          => 'audio/prs.sid',
                                  'pw'            => 'application/x-pw',
                                  'py'            => 'application/x-python',
                                  'pyc'           => 'application/x-python-bytecode',
                                  'pyo'           => 'application/x-python-bytecode',
                                  'qif'           => 'application/x-qw',
                                  'qt'            => 'video/quicktime',
                                  'qtvr'          => 'video/quicktime',
                                  'ra'            => 'audio/x-pn-realaudio',
                                  'ram'           => 'audio/x-pn-realaudio',
                                  'rar'           => 'application/x-rar',
                                  'ras'           => 'image/x-cmu-raster',
                                  'rdf'           => 'text/rdf',
                                  'rej'           => 'application/x-reject',
                                  'rgb'           => 'image/x-rgb',
                                  'rle'           => 'image/rle',
                                  'rm'            => 'audio/x-pn-realaudio',
                                  'roff'          => 'application/x-troff',
                                  'rpm'           => 'application/x-rpm',
                                  'rss'           => 'text/rss',
                                  'rtf'           => 'application/rtf',
                                  'rtx'           => 'text/richtext',
                                  's3m'           => 'audio/x-s3m',
                                  'sam'           => 'application/x-amipro',
                                  'scm'           => 'text/x-scheme',
                                  'sda'           => 'application/vnd.stardivision.draw',
                                  'sdc'           => 'application/vnd.stardivision.calc',
                                  'sdd'           => 'application/vnd.stardivision.impress',
                                  'sdp'           => 'application/vnd.stardivision.impress',
                                  'sds'           => 'application/vnd.stardivision.chart',
                                  'sdw'           => 'application/vnd.stardivision.writer',
                                  'sgi'           => 'image/x-sgi',
                                  'sgl'           => 'application/vnd.stardivision.writer',
                                  'sgm'           => 'text/sgml',
                                  'sgml'          => 'text/sgml',
                                  'sh'            => 'application/x-shellscript',
                                  'shar'          => 'application/x-shar',
                                  'shtml'         => 'text/html',
                                  'siag'          => 'application/x-siag',
                                  'sid'           => 'audio/prs.sid',
                                  'sik'           => 'application/x-trash',
                                  'silo'          => 'model/mesh',
                                  'sit'           => 'application/x-stuffit',
                                  'skd'           => 'application/x-koan',
                                  'skm'           => 'application/x-koan',
                                  'skp'           => 'application/x-koan',
                                  'skt'           => 'application/x-koan',
                                  'slk'           => 'text/spreadsheet',
                                  'smd'           => 'application/vnd.stardivision.mail',
                                  'smf'           => 'application/vnd.stardivision.math',
                                  'smi'           => 'application/smil',
                                  'smil'          => 'application/smil',
                                  'sml'           => 'application/smil',
                                  'sms'           => 'application/x-sms-rom',
                                  'snd'           => 'audio/basic',
                                  'so'            => 'application/x-sharedlib',
                                  'spd'           => 'application/x-font-speedo',
                                  'spl'           => 'application/x-futuresplash',
                                  'sql'           => 'text/x-sql',
                                  'src'           => 'application/x-wais-source',
                                  'stc'           => 'application/vnd.sun.xml.calc.template',
                                  'std'           => 'application/vnd.sun.xml.draw.template',
                                  'sti'           => 'application/vnd.sun.xml.impress.template',
                                  'stm'           => 'audio/x-stm',
                                  'stw'           => 'application/vnd.sun.xml.writer.template',
                                  'sty'           => 'text/x-tex',
                                  'sun'           => 'image/x-sun-raster',
                                  'sv4cpio'       => 'application/x-sv4cpio',
                                  'sv4crc'        => 'application/x-sv4crc',
                                  'svg'           => 'image/svg+xml',
                                  'swf'           => 'application/x-shockwave-flash',
                                  'sxc'           => 'application/vnd.sun.xml.calc',
                                  'sxd'           => 'application/vnd.sun.xml.draw',
                                  'sxg'           => 'application/vnd.sun.xml.writer.global',
                                  'sxi'           => 'application/vnd.sun.xml.impress',
                                  'sxm'           => 'application/vnd.sun.xml.math',
                                  'sxw'           => 'application/vnd.sun.xml.writer',
                                  'sylk'          => 'text/spreadsheet',
                                  't'             => 'application/x-troff',
                                  'tar'           => 'application/x-tar',
                                  'tar.Z'         => 'application/x-tarz',
                                  'tar.bz'        => 'application/x-bzip-compressed-tar',
                                  'tar.bz2'       => 'application/x-bzip-compressed-tar',
                                  'tar.gz'        => 'application/x-compressed-tar',
                                  'tar.lzo'       => 'application/x-tzo',
                                  'tcl'           => 'text/x-tcl',
                                  'tex'           => 'text/x-tex',
                                  'texi'          => 'text/x-texinfo',
                                  'texinfo'       => 'text/x-texinfo',
                                  'tga'           => 'image/x-tga',
                                  'tgz'           => 'application/x-compressed-tar',
                                  'theme'         => 'application/x-theme',
                                  'tif'           => 'image/tiff',
                                  'tiff'          => 'image/tiff',
                                  'tk'            => 'text/x-tcl',
                                  'torrent'       => 'application/x-bittorrent',
                                  'tr'            => 'application/x-troff',
                                  'ts'            => 'application/x-linguist',
                                  'tsv'           => 'text/tab-separated-values',
                                  'ttf'           => 'application/x-font-ttf',
                                  'txt'           => 'text/plain',
                                  'tzo'           => 'application/x-tzo',
                                  'ui'            => 'application/x-designer',
                                  'uil'           => 'text/x-uil',
                                  'ult'           => 'audio/x-mod',
                                  'uni'           => 'audio/x-mod',
                                  'uri'           => 'text/x-uri',
                                  'url'           => 'text/x-uri',
                                  'ustar'         => 'application/x-ustar',
                                  'vcd'           => 'application/x-cdlink',
                                  'vcf'           => 'text/x-vcalendar',
                                  'vcs'           => 'text/x-vcalendar',
                                  'vct'           => 'text/x-vcard',
                                  'vfb'           => 'text/calendar',
                                  'vob'           => 'video/mpeg',
                                  'voc'           => 'audio/x-voc',
                                  'vor'           => 'application/vnd.stardivision.writer',
                                  'vrml'          => 'model/vrml',
                                  'vsd'           => 'application/vnd.visio',
                                  'wav'           => 'audio/x-wav',
                                  'wax'           => 'audio/x-ms-wax',
                                  'wb1'           => 'application/x-quattropro',
                                  'wb2'           => 'application/x-quattropro',
                                  'wb3'           => 'application/x-quattropro',
                                  'wbmp'          => 'image/vnd.wap.wbmp',
                                  'wbxml'         => 'application/vnd.wap.wbxml',
                                  'wk1'           => 'application/vnd.lotus-1-2-3',
                                  'wk3'           => 'application/vnd.lotus-1-2-3',
                                  'wk4'           => 'application/vnd.lotus-1-2-3',
                                  'wks'           => 'application/vnd.lotus-1-2-3',
                                  'wm'            => 'video/x-ms-wm',
                                  'wma'           => 'audio/x-ms-wma',
                                  'wmd'           => 'application/x-ms-wmd',
                                  'wmf'           => 'image/x-wmf',
                                  'wml'           => 'text/vnd.wap.wml',
                                  'wmlc'          => 'application/vnd.wap.wmlc',
                                  'wmls'          => 'text/vnd.wap.wmlscript',
                                  'wmlsc'         => 'application/vnd.wap.wmlscriptc',
                                  'wmv'           => 'video/x-ms-wmv',
                                  'wmx'           => 'video/x-ms-wmx',
                                  'wmz'           => 'application/x-ms-wmz',
                                  'wpd'           => 'application/wordperfect',
                                  'wpg'           => 'application/x-wpg',
                                  'wri'           => 'application/x-mswrite',
                                  'wrl'           => 'model/vrml',
                                  'wvx'           => 'video/x-ms-wvx',
                                  'xac'           => 'application/x-gnucash',
                                  'xbel'          => 'application/x-xbel',
                                  'xbm'           => 'image/x-xbitmap',
                                  'xcf'           => 'image/x-xcf',
                                  'xcf.bz2'       => 'image/x-compressed-xcf',
                                  'xcf.gz'        => 'image/x-compressed-xcf',
                                  'xht'           => 'application/xhtml+xml',
                                  'xhtml'         => 'application/xhtml+xml',
                                  'xi'            => 'audio/x-xi',
                                  'xls'           => 'application/vnd.ms-excel',
                                  'xla'           => 'application/vnd.ms-excel',
                                  'xlc'           => 'application/vnd.ms-excel',
                                  'xld'           => 'application/vnd.ms-excel',
                                  'xll'           => 'application/vnd.ms-excel',
                                  'xlm'           => 'application/vnd.ms-excel',
                                  'xlt'           => 'application/vnd.ms-excel',
                                  'xlw'           => 'application/vnd.ms-excel',
                                  'xm'            => 'audio/x-xm',
                                  'xml'           => 'text/xml',
                                  'xpm'           => 'image/x-xpixmap',
                                  'xsl'           => 'text/x-xslt',
                                  'xslfo'         => 'text/x-xslfo',
                                  'xslt'          => 'text/x-xslt',
                                  'xwd'           => 'image/x-xwindowdump',
                                  'xyz'           => 'chemical/x-xyz',
                                  'zabw'          => 'application/x-abiword',
                                  'zip'           => 'application/zip',
                                  'zoo'           => 'application/x-zoo',
                                  '123'           => 'application/vnd.lotus-1-2-3',
                                  '669'           => 'audio/x-mod'
      );
      if (!empty($mime_extension_map))
      {
        foreach ($mime_extension_map as $key => $value)
        {
          if ($key == $ext)
          {
            return $value;
          }
        }
      }

      return "";
    }

    public static function getUserIP()
    {
      // Check for shared internet/ISP IP
      if (!empty($_SERVER['HTTP_CLIENT_IP']) && self::validateIP($_SERVER['HTTP_CLIENT_IP']))
      {
        return $_SERVER['HTTP_CLIENT_IP'];
      }

      // Check for IPs passing through proxies
      if (!empty($_SERVER['HTTP_X_FORWARDED_FOR']))
      {
        // Split the string of IPs into an array
        $ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
        foreach ($ips as $ip)
        {
          $ip = trim($ip);
          if (self::validateIP($ip))
          {
            return $ip;
          }
        }
      }

      // Return REMOTE_ADDR if none of the above methods succeeded
      if (!empty($_SERVER['REMOTE_ADDR']) && self::validateIP($_SERVER['REMOTE_ADDR']))
      {
        return $_SERVER['REMOTE_ADDR'];
      }

      // If no valid IP found, return 'UNKNOWN'
      return 'UNKNOWN';
    }

    public static function validateIP($ip)
    {
      // Validate IPv4 and IPv6 addresses
      return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6) !== false;
    }

    public static function getUserAgent()
    {
      // Check if the HTTP_USER_AGENT key exists in the $_SERVER array
      if (!empty($_SERVER['HTTP_USER_AGENT']))
      {
        return $_SERVER['HTTP_USER_AGENT'];
      }
      else
      {
        return 'User-Agent not available';
      }
    }

    public static function getBrowser($http_user_agent = null)
    {
      if (empty($http_user_agent))
      {
        $http_user_agent = self::getUserAgent();
      }
      $browser_array = array(
        'MSIE'    => 'Internet Explorer',
        'Trident' => 'Internet Explorer',
        'Firefox' => 'Mozilla Firefox',
        'Chrome'  => 'Google Chrome',
        'Safari'  => 'Safari',
        'Opera'   => 'Opera',
        'OPR'     => 'Opera',
        'Edge'    => 'Microsoft Edge'
      );

      foreach ($browser_array as $regex => $value)
      {
        if (preg_match("/$regex/i", $http_user_agent))
        {
          return $value;
        }
      }

      return 'Unknown Browser';
    }

    public static function getOS($http_user_agent = null)
    {
      if (empty($http_user_agent))
      {
        $http_user_agent = self::getUserAgent();
      }

      $os_array = array(
        '/windows nt 10/i'      => 'Windows 10',
        '/windows nt 6.3/i'     => 'Windows 8.1',
        '/windows nt 6.2/i'     => 'Windows 8',
        '/windows nt 6.1/i'     => 'Windows 7',
        '/windows nt 6.0/i'     => 'Windows Vista',
        '/windows nt 5.2/i'     => 'Windows Server 2003/XP x64',
        '/windows nt 5.1/i'     => 'Windows XP',
        '/windows xp/i'         => 'Windows XP',
        '/macintosh|mac os x/i' => 'Mac OS X',
        '/mac_powerpc/i'        => 'Mac OS 9',
        '/linux/i'              => 'Linux',
        '/ubuntu/i'             => 'Ubuntu',
        '/iphone/i'             => 'iPhone',
        '/ipod/i'               => 'iPod',
        '/ipad/i'               => 'iPad',
        '/android/i'            => 'Android',
        '/blackberry/i'         => 'BlackBerry',
        '/webos/i'              => 'Mobile'
      );

      foreach ($os_array as $regex => $value)
      {
        if (preg_match($regex, $http_user_agent))
        {
          return $value;
        }
      }

      return 'Unknown OS Platform';
    }

    /**
     * Deletes all files and subdirectories within a specified folder.
     *
     * @param   string  $folderPath  The path to the folder to be deleted.
     *
     * @return bool True on success, false on failure.
     */
    public static function deleteFolderContents($folderPath)
    {
      if (!is_dir($folderPath))
      {
        return false; // Not a directory
      }

      $items = array_diff(scandir($folderPath), array('.', '..'));

      foreach ($items as $item)
      {
        $path = $folderPath . DIRECTORY_SEPARATOR . $item;
        if (is_dir($path))
        {
          // Recursively delete subdirectories
          self::deleteFolderContents($path);
          rmdir($path); // Remove the directory itself
        }
        else
        {
          // Delete files
          unlink($path);
        }
      }

      return true;
    }

    /**
     * Method to check if System Plugin is enabled
     * @return bool
     *
     * @since 1.0.5
     */
    public static function checkSystemPlg($plg_name)
    {
      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db    = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select(['extension_id', 'enabled', 'type']);
      $query->from($db->quoteName('#__extensions'));
      $query->where($db->quoteName('name') . ' = ' . $db->quote($plg_name));
      $query->where($db->quoteName('type') . ' = ' . $db->quote('plugin'));
      $db->setQuery($query);
      $formeaSystemPlugin = $db->loadObject();
      $enabled            = false;
      if (!empty($formeaSystemPlugin))
      {
        $enabled = (int) $formeaSystemPlugin->enabled > 0;
      }

      return $enabled;
    }

  }
