<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_formea
 *
 * @copyright   Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @version     1.3.0
 */

namespace Joomla\Component\Formea\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\Database\DatabaseQuery;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Component\Formea\Administrator\Helper\FormeaHelper;
use Joomla\Component\Formea\Site\Libraries\FormeaEltype;
use Joomla\Database\DatabaseDriver;
use RuntimeException;

/**
 * Methods supporting a list of formea records.
 *
 * @since  1.0
 */
class ElementsModel extends ListModel
{
	/**
	 * @var CMSApplicationInterface|null
	 * @since 1.0.0
	 */
	protected $app;

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @see     \JControllerLegacy
	 * @since   1.0
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id',
				'a.id',
				'title',
				'a.title',
				'alias',
				'a.alias',
				'type',
				'a.type',
				'state',
				'a.state',
				'checked_out',
				'a.checked_out',
				'checked_out_time',
				'a.checked_out_time',
				'created_date',
				'a.created_date',
				'modified_date',
				'a.modified_date',
				'modified_by',
				'a.modified_by',
				'created_by',
				'a.created_by',
				'access_level',
				'a.access_level',
				'publish_up',
				'a.publish_up',
				'publish_down',
				'a.publish_down',
			];
		}

		parent::__construct($config);
		$this->app = Factory::getApplication();
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  DatabaseQuery
	 *
	 * @since   1.0
	 */
	protected function getListQuery()
	{
		// Filter on the language.
		$defaultLang = $this->app->getLanguage()->getTag();
		$language    = $this->getState('filter.language', $defaultLang);

		$columns = [
			'a.id',
			'a.title',
			'a.alias',
			'a.type',
			'a.state',
			'a.checked_out',
			'a.checked_out_time',
			'a.created_date',
			'a.modified_date',
			'a.modified_by',
			'a.created_by',
			'a.access_level',
			'a.publish_up',
			'a.publish_down'
		];
		// Create a new query object.
		/** @var DatabaseDriver $db */
		$db    = $this->getDatabase();
		$query = $db->getQuery(true);

		// Select the required fields from the table.
		$query->select(
			$db->quoteName(
				explode(
					', ',
					$this->getState(
						'list.select',
						implode(', ', $columns)
					)
				)
			)
		);

		$query->from($db->quoteName('#__formea_elements', 'a'));

		$subQuery = $db->getQuery(true);
		$subQuery->select('dt.caption')
			->from($db->quoteName('#__formea_element_details', 'dt'))
			->where(
				$db->quoteName('a.id') . ' = ' . $db->quoteName('dt.element_id') . ' AND ' .
					$db->quoteName('dt.language') . ' = ' . $db->quote($language)
			);
		$query->select('(' . $subQuery . ') AS caption');
		//$query->where($db->quoteName('dt.language') . ' = ' . $db->quote($language));

		$query->select($db->quoteName('b.title', 'access_level_title'))->innerJoin(
			$db->quoteName('#__viewlevels', 'b') . ' ON ' . $db->quoteName('a.access_level') . ' = ' .
				$db->quoteName('b.id')
		);


		// Join over the users for the checked out user.
		$query->select($db->quoteName('uc.name', 'editor'))
			->join(
				'LEFT',
				$db->quoteName('#__users', 'uc') . ' ON ' . $db->quoteName('uc.id') . ' = ' . $db->quoteName('a.checked_out')
			);


		// Filter by access level.
		if ($access = $this->getState('filter.access')) {
			$query->where($db->quoteName('a.access_level') . ' = ' . (int) $access);
		}

		// Filter by published state
		$published = (string) $this->getState('filter.published');

		if (is_numeric($published)) {
			$query->where($db->quoteName('a.state') . ' = ' . (int) $published);
		} elseif ($published === '') {
			$query->where('(' . $db->quoteName('a.state') . ' = 0 OR ' . $db->quoteName('a.state') . ' = 1)');
		}


		// Filter by search in name.
		$search = $this->getState('filter.search');

		if (!empty($search)) {
			if (stripos($search, 'id:') === 0) {
				$query->where('a.id = ' . (int) substr($search, 3));
			} else {
				$search = $db->quote('%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%'));
				$query->where(
					'(' . $db->quoteName('a.title') . ' LIKE ' . $search . ')'
				);
			}
		}


		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'a.id');
		$orderDirn = $this->state->get('list.direction', 'DESC');


		$query->order($db->escape($orderCol . ' ' . $orderDirn));

		return $query;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	protected function populateState($ordering = 'a.id', $direction = 'DESC')
	{
		$app = Factory::getApplication();

		// Adjust the context to support modal layouts.
		if ($layout = $app->input->get('layout')) {
			$this->context .= '.' . $layout;
		}


		// List state information.
		parent::populateState($ordering, $direction);
	}

	public function getItems()
	{
		// Get a storage key.
		$store = $this->getStoreId();

		// Try to load the data from internal storage.
		if (isset($this->cache[$store])) {
			return $this->cache[$store];
		}

		try {
			$elTypeConfig = [];
			// Load the list items and add the items to the internal cache.
			$items      = $this->_getList($this->_getListQuery(), $this->getStart(), $this->getState('list.limit'));
			$totalItems = count($items);
			for ($i = 0; $i < $totalItems; $i++) {
				$items[$i]->imageDetails = null;
				if (!isset($elTypeConfig[$items[$i]->type])) {
					//get new
					$elType      = new FormeaEltype(['name' => $items[$i]->type]);
					$elementType = $elType->getElementType();
					if (isset($elementType->config->imageDetails)) {
						$elTypeConfig[$items[$i]->type]  = $elementType;
						$items[$i]->imageDetails         = $elementType->config->imageDetails;
						$items[$i]->element_type_enabled = filter_var($elementType->state, FILTER_VALIDATE_BOOLEAN);
						$items[$i]->element_type_name    = $elementType->name;
						$items[$i]->element_type_title   = $elementType->title;
					}
				} else {
					$items[$i]->imageDetails         = $elTypeConfig[$items[$i]->type]->config->imageDetails;
					$items[$i]->element_type_enabled = filter_var($elTypeConfig[$items[$i]->type]->state, FILTER_VALIDATE_BOOLEAN);
					$items[$i]->element_type_name    = $elTypeConfig[$items[$i]->type]->name;
					$items[$i]->element_type_title   = $elTypeConfig[$items[$i]->type]->title;
				}
			}
			$this->cache[$store] = $items;
		} catch (RuntimeException $e) {
			$this->app->enqueueMessage($e->getMessage(), 'error');

			return false;
		}

		return $this->cache[$store];
	}

	/**
	 * Get published element types
	 * @return array|mixed
	 *
	 * @since 1.0.0
	 */
	public function getAvailableEltypes()
	{
		return FormeaHelper::getInstalledElTypes(1);
	}
}
