<?php

/**
 * @package     Joomla.Site
 * @subpackage  com_formea
 *
 * @copyright   Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @version     1.3.0
 */

namespace Joomla\Component\Formea\Site\Libraries;

defined('_JEXEC') or die;

use Exception;
use Feseur\Library\FsrDate;
use Feseur\Library\FsrFieldQuery;
use Feseur\Library\FsrHelper;
use Feseur\Library\FsrResponse;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\User;
use Joomla\Component\Formea\Site\Helper\FormeaGeneralHelper;
use Joomla\Database\DatabaseDriver;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;
use Joomla\Session\SessionInterface;
use stdClass;

class FormeaForm
{
  /**
   * @var int
   * @since 1.0.0
   */
  public $id = 0;

  /**
   * @var CMSApplicationInterface|null
   * @since 1.0.0
   */
  protected $app;

  /**
   * @var string
   * @since 1.0.0
   */
  public $langTag;

  /**
   * @var string
   * @since 1.0.0
   */
  public $langTagUnderScore;

  /**
   * @var Registry
   * @since 1.0.0
   */
  public $details;

  /**
   * @var Registry
   * @since 1.0.0
   */
  public $themes;
  public $layoutElements;
  public $elements;
  public $elementsRaw;

  /**
   * @var DatabaseDriver
   * @since 1.0.0
   */
  private $db;

  /**
   * @var User
   * @since 1.0.0
   */
  private $user;

  /**
   * @var bool
   * @since 1.0.0
   */
  private $preview = false;

  /**
   * @var FormeaTheme
   * @since 1.0.0
   */
  private $themeClass;

  /**
   * @var FormeaCore[]
   * @since 1.0.0
   */
  protected $coreClasses = [];

  private $formObject;

  public $uniqueString;

  public $formPages;

  /**
   * @var FormeaElement[]
   * @since 1.0.0
   */
  protected $elementClasses = [];

  /**
   * @var FormeaGroupedElement[]
   * @since 1.0.3
   */
  protected $groupedElementClasses = [];

  /**
   * @var FormeaGridOperations | null
   * @since 1.0.3
   */
  protected $gridOperation = null;

  public function __construct($config = [])
  {
    try {
      if (isset($config['id'])) {
        $this->id = (int) $config['id'];
      }
      if ($this->id < 1) {
        throw new Exception('No ID');
      }
      if (isset($config['preview'])) {
        $this->preview = $config['preview'];
      }
      $this->app               = Factory::getApplication();
      $this->langTag           = $this->app->getLanguage()->getTag();
      $this->langTagUnderScore = str_replace('-', '_', $this->langTag);
      $this->user              = $this->app->getIdentity();
      $container               = Factory::getContainer();
      $this->db                = $container->get('DatabaseDriver');
      $this->details           = new Registry([]);
      if (isset($config['uniqueString'])) {
        $this->uniqueString = $config['uniqueString'];
      }
      $formUniqueString = $this->getUniqueString();
      $commonConfig     = [
        'formUniqueString' => $formUniqueString,
        'formId'           => $this->id
      ];
      $this->addCoreClass('RAW', new FormeaCore($commonConfig));
    } catch (Exception $e) {
    }
  }

  public function getObject($state = -1)
  {
    if (empty($this->formObject)) {
      $db    = $this->db;
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_forms'));
      $query->where($db->quoteName('id') . ' = ' . $db->quote($this->id));
      if ($state > -1) {
        $query->where($db->quoteName('state') . ' = ' . $db->quote($state));
      }
      $db->setQuery($query);
      $form = $db->loadObject();
      if (!empty($form)) {
        $formUniqueString = $this->getUniqueString();

        $form->params   = new Registry($form->params);
        $form->honeypot = $this->getHoneyPotElement();
        $commonConfig   = [
          'formUniqueString' => $formUniqueString,
          'formId'           => $this->id,
          'formObject'       => $form
        ];
        if ($form->form_type == 1) {
          $renderer_id       = $form->params->get('renderer_id', 0);
          $renderer_style_id = $form->params->get('renderer_style_id', 0);
          if ($renderer_id > 0) {
            $rendererExtension = FormeaGeneralHelper::getCoreExtension($renderer_id, 1, 1);
            if (!empty($rendererExtension)) {
              $rendererFile      = 'Formea' . $rendererExtension->name . '.php';
              $rendererDirectory = JPATH_ROOT . '/components/com_formea/plugins/core/' . $rendererExtension->name . '/';
              if (is_file(Path::clean($rendererDirectory . $rendererFile)))
                require_once $rendererDirectory . $rendererFile;
              $coreConfig = array_merge($commonConfig, [
                'id'      => $renderer_id,
                'styleId' => $renderer_style_id,
              ]);
              $clsName    = 'Formea' . $rendererExtension->name;
              $renderer   = new $clsName($coreConfig);
              $this->addCoreClass($renderer_id, $renderer);
            }
          }
        }

        //check for cores
        $query->clear();
        $query->select(['a.*', 'b.id', 'b.extension_type']);
        $query->from($db->quoteName('#__formea_form_cores', 'a'))->innerJoin(
          $db->quoteName('#__formea_cores', 'b') . ' ON ' . $db->quoteName('a.corename') . ' = ' .
            $db->quoteName('b.name')
        );
        $query->where($db->quoteName('a.form_id') . ' = ' . $db->quote($this->id));
        $query->where($db->quoteName('b.extension_type') . ' IN (0,2)');
        $query->where($db->quoteName('a.enabled') . ' = ' . $db->quote(1));
        $db->setQuery($query);
        $cores      = $db->loadObjectList();
        $totalCores = count($cores);
        for ($i = 0; $i < $totalCores; $i++) {
          if (!empty($cores[$i]->params)) {
            $cores[$i]->params = json_decode($cores[$i]->params);
          } else {
            $cores[$i]->params = [];
          }
          $cores[$i]->params = new Registry($cores[$i]->params);
          $clsName           = 'Formea' . $cores[$i]->corename;

          if (is_file(Path::clean(FORMEA_CORE_PATH . '/' . $cores[$i]->corename . '/' . $clsName . '.php'))) {
            $extendedConfig = array_merge($commonConfig, ['formParams' => $cores[$i]->params]);
            require_once FORMEA_CORE_PATH . '/' . $cores[$i]->corename . '/' . $clsName . '.php';
            $_formeaCoreClass = new $clsName($extendedConfig);
            $this->addCoreClass($cores[$i]->id, $_formeaCoreClass);
          }
        }
      }
      $this->formObject = $form;
    }

    return $this->formObject;
  }

  public function getDetails()
  {
    if (empty($this->details->toArray())) {
      $db    = $this->db;
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_formdetails'));
      $query->where($db->quoteName('form_id') . ' = ' . $this->id);
      $db->setQuery($query);
      $details     = $db->loadObjectList();
      $detailArray = [];
      if (!empty($details)) {
        $totalDetails = count($details);
        for ($i = 0; $i < $totalDetails; $i++) {
          $lTag               = str_replace('-', '_', $details[$i]->language);
          $detailArray[$lTag] = [
            'form_heading' => $details[$i]->form_heading,
            'metakey'      => $details[$i]->metakey,
            'metadesc'     => $details[$i]->metadesc,
          ];
        }
      }
      $defaultValue = FormeaGeneralHelper::getDefaultLangValField([
        'form_heading',
        'metakey',
        'metadesc'
      ], $detailArray, $this->langTag);
      if (!empty($defaultValue)) {
        $this->details = new Registry($defaultValue);
      }
    }

    return $this->details;
  }

  public function getThemes()
  {
    if (empty($this->themes)) {
      $db    = $this->db;
      $query = $db->getQuery(true);
      $query->select(array('load_css', 'load_js', 'theme_id', 'style_id'));
      $query->from($db->quoteName('#__formea_formthemes'));
      $query->where($db->quoteName('form_id') . ' = ' . $this->id);
      $db->setQuery($query);
      $themes = $db->loadAssoc();
      if (!empty($themes)) {
        $this->themes = new Registry($themes);
      }
    }

    return $this->themes;
  }

  public function getThemeClass($config = [])
  {
    if (empty($this->themeClass)) {
      $this->getObject();
      if ($this->preview) {
        $this->themeClass = new FormeaTheme($config);
      } else {
        $formUniqueString = $this->getUniqueString();
        $theme            = $this->getThemes();
        $theme_id         = (int) $theme->get('theme_id', 0);
        $theme_style_id   = (int) $theme->get('style_id', 0);
        $coreClasses      = $this->getCoreClasses();

        $config = array_merge($config, [
          'id'                => $theme_id,
          'style_id'          => $theme_style_id,
          'langTag'           => $this->langTag,
          'langTagUnderScore' => $this->langTagUnderScore,
          'coreClasses'       => $coreClasses,
          'formUniqueString'  => $formUniqueString
        ]);
        if ($theme_id < 1) {
          $this->themeClass = new FormeaTheme($config);
        } else {
          $container = Factory::getContainer();
          /** @var DatabaseDriver $db */
          $db    = $container->get('DatabaseDriver');
          $query = $db->getQuery(true);
          $query->select('*');
          $query->from($db->quoteName('#__formea_themes'));
          $query->where($db->quoteName('id') . ' = ' . $db->quote($theme_id));
          $db->setQuery($query);
          $themeObject = $db->loadObject();
          if (empty($themeObject)) {
            $this->themeClass = new FormeaTheme($config);
          } else {
            $config    = array_merge($config, [
              'formTheme' => $themeObject
            ]);
            $themePath = JPATH_ROOT . '/components/com_formea/plugins/theme/' . $themeObject->name . '/';
            $className = 'Formea' . $themeObject->name;
            if (is_file(Path::clean($themePath . $className . '.php'))) {
              require_once $themePath . $className . '.php';
              $this->themeClass = new $className($config);
            } else {
              $this->themeClass = new FormeaTheme($config);
            }
          }
        }
      }
    }

    return $this->themeClass;
  }

  public function getGridOperation()
  {
    if (empty($this->gridOperation)) {
      $formObject  = $this->getObject();
      $themeClass  = $this->getThemeClass();
      $coreClasses = $this->getCoreClasses();

      $this->gridOperation = new FormeaGridOperations($this->id, 0, [
        'isPreview'        => $this->preview,
        'formUniqueString' => $this->getUniqueString(),
        'themeClass'       => $themeClass,
        'coreClasses'      => $coreClasses
      ]);
    }

    return $this->gridOperation;
  }

  public function renderForm($pages = [], $formObject = null)
  {
    $formUniqueString = $this->getUniqueString();
    if (empty($pages)) {
      $pages = $this->getPages();
    }
    if (empty($formObject)) {
      $formObject = $this->getObject();
    }
    $commonConfig = [
      'formUniqueString' => $formUniqueString,
      'formId'           => $this->id
    ];
    /** @var FormeaTheme $themeClass */
    $themeClass = $this->getThemeClass($commonConfig);


    //See if any core extension included
    $coreClasses = $this->getCoreClasses();
    $renderer    = $coreClasses['RAW'];
    if ($formObject->form_type == 1) {
      $renderer_id = $formObject->params->get('renderer_id', 0);
      if ($renderer_id > 0) {
        if (isset($coreClasses[$renderer_id])) {
          $renderer = $coreClasses[$renderer_id];
        } else {
          $rendererExtension = FormeaGeneralHelper::getCoreExtension($renderer_id, 1, 1);
          if (!empty($rendererExtension)) {
            $rendererFile      = 'Formea' . $rendererExtension->name . '.php';
            $rendererDirectory = JPATH_ROOT . '/components/com_formea/plugins/core/' . $rendererExtension->name . '/';
            if (is_file(Path::clean($rendererDirectory . $rendererFile)))
              require_once $rendererDirectory . $rendererFile;
            $clsName  = 'Formea' . $rendererExtension->name;
            $renderer = new $clsName($commonConfig);
            $this->addCoreClass($renderer_id, $renderer);
          }
        }
      }
    }

    $themeClass->setRenderer($renderer);

    $html                = '';
    $totalPages          = count($pages);
    $groupElementClasses = $this->getGroupElementClasses();
    for ($i = 0; $i < $totalPages; $i++) {

      $pageContent = $themeClass->drawPage($this->id, $pages[$i], $i, $totalPages);
      $pageLayout  = $this->drawHTML($pages[$i], $pages[$i]->rows, $themeClass, $groupElementClasses);
      $html        .= str_replace('[LAYOUT_' . $pages[$i]->id . ']', $pageLayout, $pageContent);
    }
    $cores      = $this->getCoreClasses();
    $cores      = array_values($cores);
    $totalCores = count($cores);

    for ($j = 0; $j < $totalCores; $j++) {
      $html = $cores[$j]->onBeforeDisplayHtml($html);
    }

    return $html;
  }

  /**
   * @param   stdClass                $page
   * @param   FormeaGridItem[]        $items
   * @param   FormeaTheme             $themeClass
   * @param   FormeaGroupedElement[]  $groupElementClasses
   *
   * @since 1.0.0
   */
  public function drawHTML($page, $items, $themeClass, $groupElementClasses)
  {

    $html       = '';
    $totalItems = count($items);
    $key        = '';
    for ($i = 0; $i < $totalItems; $i++) {
      if ($items[$i]->item_type == 0) {
        //
        $mainContent = $themeClass->drawRow($page, $items[$i]);
        $key         = '[COLUMN_' . $page->id . '_' . $items[$i]->id . ']';
      }
      if ($items[$i]->item_type == 1) {
        //
        $mainContent = $themeClass->drawColumn($page, $items[$i]);
        $key         = '[ITEM_' . $page->id . '_' . $items[$i]->id . ']';
      }
      if ($items[$i]->item_type == 2) {
        if ($items[$i]->state > 0) {
          $mainContent = $themeClass->drawItem($page, $items[$i]);
        } else {
          $mainContent = '';
        }
      }
      if ($items[$i]->item_type == 3) {
        if ($items[$i]->state > 0) {
          $groupedElementClass = null;
          if (isset($groupElementClasses[$items[$i]->group_id])) {
            $groupedElementClass = $groupElementClasses[$items[$i]->group_id];
          }
          //Grouped Element
          $mainContent = $themeClass->drawGroupBlock($this->id, $page, $items[$i], $groupedElementClass);
          $key         = '[BLOCK_ITEM_' . $page->id . '_' . $items[$i]->id . ']';
          if ($items[$i]->group_type == 1) {
            $groupedEl = $themeClass->renderInputGroupField($page, $items[$i], $this->langTagUnderScore, $groupedElementClass);

            $mainContent = str_replace($key, $groupedEl, $mainContent);
            $key         = '';
          }
        } else {
          $mainContent = '';
        }
      }

      if (!empty($items[$i]->items)) {
        $content = $this->drawHTML($page, $items[$i]->items, $themeClass, $groupElementClasses);
        if (empty($content)) {
          $content = '';
        }
        $mainContent = str_replace($key, $content, $mainContent);
      } else {
        $mainContent = str_replace($key, '', $mainContent);
      }
      if (empty($mainContent)) {
        $mainContent = '';
      }
      $html .= $mainContent;
    }

    return $html;
  }

  /**
   * @return array|mixed
   *
   * @deprecated 1.0.3
   * @since      version
   */
  public function _getLayoutData()
  {
    if (empty($this->layoutElements)) {
      $elements = $this->getElementsRaw();

      if (!empty($elements)) {
        $totalElements = count($elements);
        for ($i = 0; $i < $totalElements; $i++) {

          $element       = new FormeaElement(['element_id' => $elements[$i]->element_id, 'form_id' => $this->id]);
          $labelSettings = $elements[$i]->labelSettings;
          $defaultLabel  = null;
          if (!empty($labelSettings)) {
            $labelSettings = json_decode($labelSettings);
            $totalLabel    = count($labelSettings);
            for ($j = 0; $j < $totalLabel; $j++) {
              if ($labelSettings[$j]->lang_code === $this->langTagUnderScore) {
                $defaultLabel               = $labelSettings[$j]->setting;
                $defaultLabel->helper_align = $labelSettings[$j]->helper_align;
                break;
              }
            }
            if (empty($defaultLabel)) {
              $defaultLabel               = $labelSettings[0]->setting;
              $defaultLabel->helper_align = $labelSettings[0]->helper_align;
            }
          }
          $columnAttributes = [];
          if (!empty($elements[$i]->column_class)) {
            $columnAttributes['class'] = $elements[$i]->column_class;
          }
          $elements[$i]->labelSettings = new Registry($defaultLabel);
          $elements[$i]->input         = $element->renderElement($columnAttributes, $elements[$i]->labelSettings);
          $details                     = $element->getDetails(true, true);
          $defaultDetail               = FormeaGeneralHelper::getDefaultLangValField([
            'caption',
            'placeholder',
            'description'
          ], $details, $this->langTag);
          $elements[$i]->details       = new Registry($defaultDetail);
        }
        $this->layoutElements = $elements;
      }
    }

    return $this->layoutElements;
  }

  /**
   * @param   FsrFieldQuery[]  $whereClause
   *
   * @return array|mixed
   *
   * @since 1.0.0
   */
  public function getElementsRaw($preview = false, $whereClause = [])
  {
    if (empty($this->elementsRaw)) {
      $operator          = $this->getGridOperation();
      $this->elementsRaw = $operator->getElementsRaw($whereClause);
    }

    return $this->elementsRaw;
  }

  public function getElements($whereClause = [])
  {
    if (empty($this->elements)) {
      $operator       = $this->getGridOperation();
      $this->elements = $operator->getElements($whereClause);
    }

    return $this->elements;
  }

  public function getGroupElement($group_id, $setIndex = 0)
  {
    $operator = $this->getGridOperation();
    $operator->set('setIndex', $setIndex);
    $singleGroupedElementArray = $operator->getSingleGroupElement($group_id, $setIndex);
    $html                      = '';
    $scripts                   = [];
    if (isset($singleGroupedElementArray['groupElement'])) {
      $themeClass = $this->getThemeClass();
      $dummyPage  = [
        'id'         => -1,
        'form_id'    => $this->id,
        'title'      => '',
        'subtitle'   => '',
        'icon'       => '',
        'settings'   => [],
        'layout'     => '',
        'styleSheet' => '',
      ];
      /** @var FormeaGroupedElement $groupElementClass */
      $groupElementClass = $singleGroupedElementArray['groupedElementClass'];
      $groupElementClass->set('setIndex', $setIndex);
      $html = $this->drawHTML((object) $dummyPage, [$singleGroupedElementArray['groupElement']], $themeClass, [$group_id => $groupElementClass]);
      //$elementClasses = [];

      $elementClasses = $groupElementClass->getElementClasses();
      /** @var FormeaELement[] $elementClasses */

      $elementClasses      = array_values($elementClasses);
      $totalElementClasses = count($elementClasses);
      for ($i = 0; $i < $totalElementClasses; $i++) {
        $elementClasses[$i]->set('group_id', $group_id);
        $elementClasses[$i]->set('isGroup', true);
        $elementClasses[$i]->set('setIndex', $setIndex);
        $_script = $elementClasses[$i]->getGeneratedScript();
        if (!empty($_script)) {
          $scripts[] = $_script;
        }
      }
    }

    return [
      'html'    => $html,
      'scripts' => $scripts
    ];
  }

  public function getCoreClasses($whereClause = [])
  {
    return $this->coreClasses;
  }

  public function setCoreProperty($prop, $val)
  {
    $coreClasses = $this->getCoreClasses();
    foreach ($coreClasses as $class) {
      $class->set($prop, $val);
    }
  }

  /**
   * @param   FormeaCore  $core
   *
   *
   * @since 1.0.0
   */
  public function addCoreClass($key, $core)
  {
    $this->coreClasses[$key] = $core;
  }

  public function getElementClasses($whereClause = [])
  {
    $operator = $this->getGridOperation();

    return $operator->getElementClasses();
  }

  public function addElementClass($element_id, $formeaElement)
  {
    $this->elementClasses[$element_id] = $formeaElement;
  }

  /**
   * Get the JS/CSS required or generated by the Element
   *
   * @since 1.0.0
   */
  public function getElementsAssets()
  {
    $ret = [
      'inlineScripts' => [],
      'inlineStyles'  => [],
      'fileScripts'   => [],
      'fileStyles'    => []
    ];

    $pages              = $this->getPages();
    $allItemStyleSheets = [];
    $totalPages         = count($pages);
    for ($j = 0; $j < $totalPages; $j++) {
      $allItemStyleSheets = $this->getAllItemsStyleSheets($pages[$j]->rows, $allItemStyleSheets);
    }
    if (!empty($allItemStyleSheets)) {
      $ret['inlineStyles'] = $allItemStyleSheets;
    }


    $elementClasses = $this->getElementClasses();

    /** @var FormeaElement[] $elementClasses */
    $elementClasses      = array_values($elementClasses);
    $totalElementClasses = count($elementClasses);
    for ($i = 0; $i < $totalElementClasses; $i++) {
      $inlineScript = $elementClasses[$i]->getGeneratedScript();
      if (!empty($inlineScript)) {
        $ret['inlineScripts'][] = $inlineScript;
      }

      $inlineStyle = $elementClasses[$i]->getGeneratedStyle();
      if (!empty($inlineStyle)) {
        $ret['inlineStyles'][] = $inlineStyle;
      }

      $fileScripts = $elementClasses[$i]->getLoadableScript();
      if (!empty($fileScripts)) {
        $ret['fileScripts'] = array_merge($ret['fileScripts'], $fileScripts);
      }
      $fileStyles = $elementClasses[$i]->getLoadableStyle();
      if (!empty($fileStyles)) {
        $ret['fileStyles'] = array_merge($ret['fileStyles'], $fileStyles);
      }
    }

    return (object) $ret;
  }

  /**
   * Get the JS/CSS required or generated by the Core Extensions
   *
   * @since 1.0.0
   */
  public function getCoreAssets()
  {
    $ret = [
      'inlineScripts' => [],
      'inlineStyles'  => [],
      'fileScripts'   => [],
      'fileStyles'    => []
    ];


    $coreClasses = $this->getCoreClasses();

    /** @var FormeaCore[] $coreClasses */
    $elementClasses      = array_values($coreClasses);
    $totalElementClasses = count($elementClasses);
    for ($i = 0; $i < $totalElementClasses; $i++) {
      $inlineScript = $elementClasses[$i]->getGeneratedScript();
      if (!empty($inlineScript)) {
        $ret['inlineScripts'][] = $inlineScript;
      }

      $inlineStyle = $elementClasses[$i]->getGeneratedStyle();
      if (!empty($inlineStyle)) {
        $ret['inlineStyles'][] = $inlineStyle;
      }

      $fileScripts = $elementClasses[$i]->getLoadableScript();
      if (!empty($fileScripts)) {
        $ret['fileScripts'] = array_merge($ret['fileScripts'], $fileScripts);
      }
      $fileStyles = $elementClasses[$i]->getLoadableStyle();
      if (!empty($fileStyles)) {
        $ret['fileStyles'] = array_merge($ret['fileStyles'], $fileStyles);
      }
    }

    $groupedElementClasses = $this->getGroupElementClasses();
    if (!empty($groupedElementClasses)) {
      $groupedElementClasses      = array_values($groupedElementClasses);
      $totalGroupedElementClasses = count($groupedElementClasses);
      $groupCores                 = [];
      for ($j = 0; $j < $totalGroupedElementClasses; $j++) {
        $_groupcores = $groupedElementClasses[$j]->groupCores;
        if (!empty($_groupcores)) {
          $groupCores = array_merge($_groupcores, $groupCores);
        }
      }
      if (!empty($groupCores)) {
        $groupCores      = array_values($groupCores);
        $totalGroupCores = count($groupCores);
        for ($k = 0; $k < $totalGroupCores; $k++) {
          $inlineScript = $groupCores[$k]->getGeneratedScript();
          if (!empty($inlineScript)) {
            $ret['inlineScripts'][] = $inlineScript;
          }

          $inlineStyle = $groupCores[$k]->getGeneratedStyle();
          if (!empty($inlineStyle)) {
            $ret['inlineStyles'][] = $inlineStyle;
          }

          $fileScripts = $groupCores[$k]->getLoadableScript();
          if (!empty($fileScripts)) {
            $ret['fileScripts'] = array_merge($ret['fileScripts'], $fileScripts);
          }
          $fileStyles = $groupCores[$k]->getLoadableStyle();
          if (!empty($fileStyles)) {
            $ret['fileStyles'] = array_merge($ret['fileStyles'], $fileStyles);
          }
        }
      }
    }

    return (object) $ret;
  }

  /**
   * Get the JS/CSS required or generated by the Theme
   *
   * @since 1.0.0
   */
  public function getThemeAssets()
  {
    $ret = [
      'inlineScripts' => [],
      'inlineStyles'  => [],
      'fileScripts'   => [],
      'fileStyles'    => []
    ];


    $themeClass   = $this->getThemeClass();
    $inlineScript = $themeClass->getGeneratedScript();
    $formObject   = $this->getObject();

    if (!empty($inlineScript)) {
      $ret['inlineScripts'][] = $inlineScript;
    }

    $inlineStyle = $themeClass->getGeneratedStyle();
    if (!empty($inlineStyle)) {
      $ret['inlineStyles'][] = $inlineStyle;
    }

    if ((int) $formObject->themes->get('load_js', 1) > 0) {
      $fileScripts = $themeClass->getLoadableScript();
      if (!empty($fileScripts)) {
        $ret['fileScripts'] = array_merge($ret['fileScripts'], $fileScripts);
      }
    }

    if ((int) $formObject->themes->get('load_css', 1) > 0) {
      $fileStyles = $themeClass->getLoadableStyle();
      if (!empty($fileStyles)) {
        $ret['fileStyles'] = array_merge($ret['fileStyles'], $fileStyles);
      }
    }


    return (object) $ret;
  }

  /**
   * Method to get all css/js required for the form
   * @return object
   *
   * @since 1.0.0
   */
  public function getAllAssets()
  {
    $ret        = [
      'inlineScripts' => [],
      'inlineStyles'  => [],
      'fileScripts'   => ['com_formea.site.general'],
      'fileStyles'    => ['com_formea.frontend-formea']
    ];
    $formObject = $this->getObject();

    if (!empty($formObject)) {
      $uniqueString = $this->getUniqueString();
      $replaceAble  = [
        '[FORMEA_TOKEN]' => $uniqueString,
        '[FORMEA_ID]'    => $this->id
      ];
      if (!empty($formObject->inline_js)) {
        $_inlineJs = $formObject->inline_js;
        foreach ($replaceAble as $key => $value) {
          $_inlineJs = str_replace($key, $value, $_inlineJs);
        }
        $ret['inlineScripts'][] = $_inlineJs;
      }
      if (!empty($formObject->inline_css)) {
        $_inlineCss = $formObject->inline_css;
        foreach ($replaceAble as $key => $value) {
          $_inlineCss = str_replace($key, $value, $_inlineCss);
        }
        $ret['inlineStyles'][] = $_inlineCss;
      }
    }


    $themeAssets   = $this->getThemeAssets();
    $coreAssets    = $this->getCoreAssets();
    $elementAssets = $this->getElementsAssets();

    foreach ($ret as $key => $arr) {
      if (!empty($themeAssets->{$key})) {
        $ret[$key] = array_merge($ret[$key], $themeAssets->{$key});
      }
      if (!empty($coreAssets->{$key})) {
        $ret[$key] = array_merge($ret[$key], $coreAssets->{$key});
      }
      if (!empty($elementAssets->{$key})) {
        $ret[$key] = array_merge($ret[$key], $elementAssets->{$key});
      }
    }

    return (object) $ret;
  }

  /**
   * @param   FormeaGridItem[]  $items
   * @param   string[]          $styleArray
   *
   * @since version
   */
  public function getAllItemsStyleSheets($items, $styleArray = [])
  {
    $totalItems = count($items);
    for ($i = 0; $i < $totalItems; $i++) {
      if (!empty($items[$i]->styleSheet)) {
        $styleArray[] = $items[$i]->styleSheet;
      }
      if (!empty($items[$i]->items)) {
        $styleArray = $this->getAllItemsStyleSheets($items[$i]->items, $styleArray);
      }
    }

    return $styleArray;
  }

  public function getSubmissionMessages()
  {
    $db    = $this->db;
    $query = $db->getQuery(true);
    $query->select('*');
    $query->from($db->quoteName('#__formea_formsubmissions'));
    $query->where($db->quoteName('form_id') . ' = ' . $db->quote($this->id));
    if (!empty($this->langTag)) {
      $query->where($db->quoteName('language') . ' = ' . $db->quote($this->langTag));
    } else {
      $query->order('id DESC');
      $query->setLimit(1, 0);
    }

    $db->setQuery($query);

    $result =  $db->loadAssoc();
    if (!empty($result)) {
      $formObject = $this->getObject();
      $result['message_type'] = $formObject->params->get('message_type', 'joomla');
    }
    return (object)$result;
  }

  /**
   * @param $submission_id
   *
   * @return FsrResponse
   *
   * @deprecated to be removed on version 1.0.4
   * @since      1.0.0
   */
  public function sendEmail($submission_id = 0)
  {
    $formeaSubmission = new FormeaSubmission($submission_id, ['langTag' => $this->langTag]);

    return $formeaSubmission->sendMail();
  }

  /**
   * @param $fieldTypes
   *
   * @Deprecated
   * @since version
   */
  public function _getElements($fieldTypes = ['*'])
  {
    $ids              = [];
    $rawElements      = $this->getElementsRaw();
    $totalRawElements = count($rawElements);
    for ($i = 0; $i < $totalRawElements; $i++) {
      $ids[] = $rawElements[$i]->element_id;
    }
    $formeaElements = new FormeaElements();
    /** @var FsrFieldQuery[] $whereClause */
    $whereClause = [
      (object) ['field' => 'id', 'operator' => 'IN', 'value' => '(' . implode(',', $ids) . ')']
    ];
    $elements    = $formeaElements->get(false, $fieldTypes, null, ['a.id' => 'DESC'], $whereClause);
  }

  /**
   * @param $submission_id
   *
   * @return array|mixed
   *
   * @deprecated To be removed on version 1.0.4
   * @since      1.0.0
   */
  public function getSubmissionData($submission_id)
  {
    $db    = $this->db;
    $query = $db->getQuery(true);
    $query->select('*');
    $query->from($db->quoteName('#__formea_submission_data'));
    $query->where($db->quoteName('form_id') . ' = ' . $db->quote($this->id));
    $query->where($db->quoteName('submission_id') . ' = ' . $db->quote($submission_id));
    $db->setQuery($query);

    return $db->loadObjectList();
  }

  public function getPages($settingRegistry = true)
  {
    if (empty($this->formPages)) {
      $db    = $this->db;
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_form_fieldsets'));
      $query->where($db->quoteName('form_id') . ' = ' . $db->quote($this->id));
      $query->order('id ASC');
      $db->setQuery($query);
      $pages = $db->loadObjectList();
      if (!empty($pages)) {
        $totalPages = count($pages);
        $rows       = $this->getRows($settingRegistry);
        $columns    = $this->getColumns($settingRegistry);
        $elements   = $this->getGridItem($settingRegistry);
        $groups     = $this->getGroupItems($settingRegistry);


        /** @var SessionInterface $session */
        $session       = $this->app->getSession();
        $submittedData = $session->get('FORMEA_FORM_SUBMISSION_' . $this->id);
        $assignedData  = [];
        if (!empty($submittedData)) {
          //sort submitted data by page
          $totalAssignedData = count($submittedData->elements);
          for ($j = 0; $j < $totalAssignedData; $j++) {
            $assignedData[$submittedData->elements[$j]->page_id][] = $submittedData->elements[$j];
          }
        }

        for ($i = 0; $i < $totalPages; $i++) {
          if (empty($pages[$i]->title)) {
            $pages[$i]->title = '';
          }
          $settings = [];
          if (!empty($pages[$i]->settings)) {
            $settings = json_decode($pages[$i]->settings);
          }
          $settings            = new Registry($settings);
          $pages[$i]->settings = $settings;
          if (!$settingRegistry) {
            $pages[$i]->settings = $settings->toArray();
          }
          $layout = '';
          if (!empty($pages[$i]->layout)) {
            $layout = $pages[$i]->layout;
          }
          $pages[$i]->layout = json_decode($layout);
          if (isset($assignedData[$pages[$i]->id])) {
            //insert extra element if required
          }

          $pages[$i]->rows = [];

          $items = $this->getFormeaGridItem($pages[$i]->id, $pages[$i]->layout, $rows, $columns, $elements, $groups);
          if (!empty($items)) {
            $pages[$i]->rows = $items;
          } else {
            $pages[$i]->rows = [];
          }


          if ($i == 0) {
            $pages[$i]->selected = true;
          } else {
            $pages[$i]->selected = false;
          }
        }
      }
      $this->formPages = $pages;
      $totalPages      = count($pages);
      $this->setCoreProperty('totalPages', $totalPages);
    }

    return $this->formPages;
  }

  public function getFormeaGridItem($page_id, $items, $rows, $columns, $elements, $groups)
  {
    $structure = [];
    if (!is_null($items)) {
      $totalItems = count($items);
      for ($i = 0; $i < $totalItems; $i++) {
        $sets = [];
        if ($items[$i]->item_type == 0) {
          //check rows
          if (isset($rows[$items[$i]->item_id])) {
            $structure[$i] = $rows[$items[$i]->item_id];
          }
        } else if ($items[$i]->item_type == 1) {
          //columns
          if (isset($columns[$items[$i]->item_id])) {
            $structure[$i] = $columns[$items[$i]->item_id];
          }
        } else if ($items[$i]->item_type == 2) {
          //element
          if (isset($elements[$items[$i]->item_id])) {
            $structure[$i] = $elements[$items[$i]->item_id];
          }
        } else {
          //group
          if (isset($groups[$items[$i]->item_id])) {

            $pushedSets     = [];
            $inputGroupItem = $groups[$items[$i]->item_id]->inputGroupItem;

            $totalSets = count($items[$i]->sets);
            for ($j = 0; $j < $totalSets; $j++) {

              if ($j == 0) {
                $pushedSets[] = $inputGroupItem;
              } else {
                $_groupedItem = new FormeaGroupedElement($groups[$items[$i]->item_id]->group_id, [
                  'isPreview'    => $this->preview,
                  'themeClass'   => $this->getThemeClass(),
                  'uniqueString' => $this->getUniqueString(),
                  'setIndex'     => $j
                ]);
                $pushedSets[] = $_groupedItem->getInputGroupParams();
              }
            }
            $sets          = $pushedSets;
            $structure[$i] = $groups[$items[$i]->item_id];
          }
        }

        if (!empty($items[$i]->items)) {
          $structure[$i]->items = $this->getFormeaGridItem($page_id, $items[$i]->items, $rows, $columns, $elements, $groups);
        }

        if (isset($structure[$i]) && !empty($structure[$i])) {
          $structure[$i]->sets = [];

          if (!empty($sets)) {
            $structure[$i]->sets = $sets;
          }
        }
      }
    }

    return $structure;
  }

  public function getGroupItems($settingRegistry, $page_id = -1)
  {
    $operator = $this->getGridOperation();
    if ($page_id > -1) {
      $operator->set('page_id', $page_id);
    }
    $operator->set('settingRegistry', $settingRegistry);

    return $operator->getGroupElements();
  }

  public function getGroupElements($settingRegistry, $page_id = -1)
  {
    $operator = $this->getGridOperation();
    if ($page_id > -1) {
      $operator->set('page_id', $page_id);
    }
    $operator->set('settingRegistry', $settingRegistry);

    return $operator->getGrouped();
  }

  public function getGroupElementClasses()
  {
    $operator = $this->getGridOperation();

    return $operator->getGroupElementClass();
  }

  public function getRows($settingRegistry = true, $page_id = -1)
  {
    $operator = $this->getGridOperation();
    if ($page_id > -1) {
      $operator->set('page_id', $page_id);
    }
    $operator->set('settingRegistry', $settingRegistry);

    return $operator->getRows();
  }

  public function getColumns($settingRegistry = true, $page_id = -1)
  {
    $operator = $this->getGridOperation();
    if ($page_id > -1) {
      $operator->set('page_id', $page_id);
    }
    $operator->set('settingRegistry', $settingRegistry);

    return $operator->getColumns();
  }

  public function getGridItem($settingRegistry = true, $page_id = -1, $preview = true)
  {
    $operator = $this->getGridOperation();
    if ($page_id > -1) {
      $operator->set('page_id', $page_id);
    }
    $operator->set('settingRegistry', $settingRegistry);
    $gridItems = $operator->getGridItem();
    //update themeClasses
    $operatorThemeClass = $operator->getElementClasses();
    foreach ($operatorThemeClass as $element_id => $elementClass) {
      $this->addElementClass($element_id, $elementClass);
    }

    return $gridItems;
  }

  public function getUniqueString()
  {
    if (empty($this->uniqueString)) {
      $this->uniqueString = FsrHelper::generateRandomUniqueString(8);
    }

    return $this->uniqueString;
  }

  public function validatePage($pageIndex, $data, $uniqueString)
  {
    $retObject = new FsrResponse();
    try {
      //get all pages first
      $db    = $this->db;
      $query = $db->getQuery(true);
      $query->select('*');
      $query->from($db->quoteName('#__formea_form_fieldsets'));
      $query->where($db->quoteName('form_id') . ' = ' . $db->quote($this->id));
      $query->order('id ASC');
      $db->setQuery($query);
      $pages = $db->loadObjectList();
      if (!isset($pages[$pageIndex])) {
        throw new Exception(Text::_('COM_FORMEA_SECTION_NOT_FOUND'));
      }
      $page_id       = $pages[$pageIndex]->id;
      $whereClause   = [];
      $whereClause[] = (object) ['field' => 'a.page_id', 'operator' => '=', 'value' => (string) $page_id];
      $_elements     = $this->getElements($whereClause);

      $elements      = [];
      $totalElements = count($_elements);
      if ($totalElements < 1) {
        throw new Exception(Text::_('COM_FORMEA_EMPTY_SUBMISSION'));
      }
      $formObject        = $this->getObject();
      $coreClasses       = $this->getCoreClasses();
      $formeaSubmissions = new FormeaSubmissions([
        'coreClasses' => $coreClasses,
        'formObject'  => $formObject
      ]);
      $assignedElements  = $formeaSubmissions->prepareForValidations($_elements, $data, $uniqueString);

      /** @var FormeaTheme $themeClass */
      $themeClass = $this->getThemeClass();
      //Get the general messages for validation
      $submissionMsgs            = $this->getSubmissionMessages();
      $hasGlobalError            = false;
      $submission                = new stdClass();
      $submission->form_id       = $this->id;
      $submission->page_index    = $pageIndex;
      $submission->page_id       = $page_id;
      $submission->classes       = ['page_feedback_element' => $themeClass->getFeedbackPageClass(false)];
      $submission->error_classes = ['page_feedback_element' => $themeClass->getFeedbackPageClass(true)];

      $totalAssignedElements = count($assignedElements);
      for ($k = 0; $k < $totalAssignedElements; $k++) {
        $formeaElement                       = new FormeaElement([
          'element_id'       => $assignedElements[$k]->element_id,
          'is_preview'       => false,
          'isPreview'        => false,
          'formUniqueString' => $uniqueString,
        ]);
        $assignedElements[$k]->error_classes = [
          'element_container'  => $themeClass->getContainerClass(true),
          'input'              => $themeClass->getErrorClass(),
          'feedback_container' => $themeClass->getFeedbackContainerClass(true),
        ];
        $assignedElements[$k]->classes       = [
          'element_container'  => $themeClass->getContainerClass(false),
          'input'              => $themeClass->getNoErrorClass(),
          'feedback_container' => $themeClass->getFeedbackContainerClass(false),
        ];
        if ($assignedElements[$k]->byPassValidation) {
          $assignedElements[$k]->validation_error  = false;
          $assignedElements[$k]->validation_result = null;
        } else {
          $assignedElements[$k] = $formeaElement->beforeValidatingSubmission($assignedElements[$k]);
          $validate             = $formeaElement->validateSubmission($assignedElements[$k]->submitted_value, $assignedElements[$k]);
          if (!$validate->success) {
            throw new Exception(Text::_('COM_FORMEA_VALIDATION_PROCESS_ERROR'));
          }

          $assignedElements[$k]->validation_error  = false;
          $assignedElements[$k]->validation_result = $validate;


          $totalValidationResults = count($validate->result);
          for ($j = 0; $j < $totalValidationResults; $j++) {
            if ($validate->result[$j]->result->result->hasError) {
              $hasGlobalError                         = true;
              $assignedElements[$k]->validation_error = true;
            }
          }
        }
        unset($assignedElements[$k]->labelSettings);
      }

      if ($hasGlobalError) {
        $msg = Text::_('COM_FORMEA_VALIDATION_GLOBAL_FORM_ERROR');
        if (isset($submissionMsgs->global_error_msg)) {
          if (!empty($submissionMsgs->global_error_msg)) {
            $msg = Text::_($submissionMsgs->global_error_msg);
          } else {
            $submissionMsgs->global_error_msg = $msg;
          }
        }
        $retObject->setMsg($msg, 'danger', true);
      }

      $retObject->setSuccess(!$hasGlobalError);
      $retObject->result = (object) [
        'hasError'       => $hasGlobalError,
        'elements'       => $assignedElements,
        'submission'     => $submission,
        'submissionMsgs' => $submissionMsgs
      ];
    } catch (Exception $e) {
      $retObject->setMsg($e->getMessage());
    }

    return $retObject;
  }

  public function validateForm($data, $uniqueString, $submittedUrl)
  {
    $retObject = new FsrResponse();
    try {

      $_elements = $this->getElements();

      $totalElements = count($_elements);
      if ($totalElements < 1) {
        throw new Exception(Text::_('COM_FORMEA_EMPTY_SUBMISSION'));
      }
      $formObject        = $this->getObject();
      $coreClasses       = $this->getCoreClasses();
      $formeaSubmissions = new FormeaSubmissions([
        'coreClasses' => $coreClasses,
        'formObject'  => $formObject
      ]);
      $assignedElements  = $formeaSubmissions->prepareForValidations($_elements, $data, $uniqueString);

      /** @var FormeaTheme $themeClass */
      $themeClass = $this->getThemeClass();
      //Get the general messages for validation
      $submissionMsgs            = $this->getSubmissionMessages();
      $hasGlobalError            = false;
      $submission                = new stdClass();
      $submission->form_id       = $this->id;
      $submission->classes       = ['page_feedback_element' => $themeClass->getFeedbackPageClass(false)];
      $submission->error_classes = ['page_feedback_element' => $themeClass->getFeedbackPageClass(true)];

      $totalAssignedElements = count($assignedElements);
      for ($k = 0; $k < $totalAssignedElements; $k++) {
        $formeaElement                       = new FormeaElement([
          'element_id'       => $assignedElements[$k]->element_id,
          'is_preview'       => false,
          'isPreview'        => false,
          'formUniqueString' => $uniqueString,
        ]);
        $assignedElements[$k]->error_classes = [
          'element_container'  => $themeClass->getContainerClass(true),
          'input'              => $themeClass->getErrorClass(),
          'feedback_container' => $themeClass->getFeedbackContainerClass(true),
        ];
        $assignedElements[$k]->classes       = [
          'element_container'  => $themeClass->getContainerClass(false),
          'input'              => $themeClass->getNoErrorClass(),
          'feedback_container' => $themeClass->getFeedbackContainerClass(false),
        ];
        if ($assignedElements[$k]->byPassValidation) {
          $assignedElements[$k]->validation_error  = false;
          $assignedElements[$k]->validation_result = null;
        } else {
          $assignedElements[$k] = $formeaElement->beforeValidatingSubmission($assignedElements[$k]);
          $validate             = $formeaElement->validateSubmission($assignedElements[$k]->submitted_value, $assignedElements[$k]);
          if (!$validate->success) {
            throw new Exception(Text::_('COM_FORMEA_VALIDATION_PROCESS_ERROR'));
          }

          $assignedElements[$k]->validation_error  = false;
          $assignedElements[$k]->validation_result = $validate;


          $totalValidationResults = count($validate->result);
          for ($j = 0; $j < $totalValidationResults; $j++) {
            if ($validate->result[$j]->result->result->hasError) {
              $hasGlobalError                         = true;
              $assignedElements[$k]->validation_error = true;
            }
          }
        }
        unset($assignedElements[$k]->labelSettings);
      }

      if ($hasGlobalError) {
        $msg = Text::_('COM_FORMEA_VALIDATION_GLOBAL_FORM_ERROR');
        if (isset($submissionMsgs->global_error_msg)) {
          if (!empty($submissionMsgs->global_error_msg)) {
            $msg = Text::_($submissionMsgs->global_error_msg);
          } else {
            $submissionMsgs->global_error_msg = $msg;
          }
        }
        $retObject->setMsg($msg, 'danger', true);
      }
      $afterSubmission = $formObject->params->get('after_submission', 'none');

      $retObject->setSuccess(!$hasGlobalError);
      $submissionResponse = new FormeaSubmissionResponse([
        'url'            => $submittedUrl,
        'hasError'       => $hasGlobalError,
        'elements'       => $assignedElements,
        'submission'     => $submission,
        'submissionMsgs' => $submissionMsgs,
        'redirect'       => false,
        'redirectUrl'    => "",
        'afterSubmission' => $afterSubmission
      ]);
      $retObject->result = $submissionResponse;
    } catch (Exception $e) {
      $retObject->setMsg($e->getMessage());
    }

    return $retObject;
  }

  public function processSubmission($data, $form_id, $uniqueString, $submitted_url = '')
  {
    $retObject = new FsrResponse();
    try {
      if (empty($data)) {
        throw new Exception(Text::_('COM_FORMEA_EMPTY_SUBMISSION'));
      }
      $unknown_submitted_url = false;
      if (empty($submitted_url)) {
        $submitted_url         = Uri::root();
        $unknown_submitted_url = true;
      }
      $redirect    = false;
      $redirectUrl = '';
      $formObject  = $this->getObject();
      /** @var FormeaTheme $themeClass */
      $themeClass = $this->getThemeClass();
      //Get the general messages for validation
      $submissionMsgs = $this->getSubmissionMessages();

      $hasGlobalError         = false;
      $submission             = new stdClass();
      $submission->form_id    = $this->id;
      $submission_data_values = [];

      $_elements = $this->getElements();


      $totalElements = count($_elements);
      if ($totalElements < 1) {
        throw new Exception(Text::_('COM_FORMEA_EMPTY_SUBMISSION'));
      }
      /** @var FormeaElement $processedElementClass */
      $processedElementClass = [];
      $coreClasses           = $this->getCoreClasses();

      $_coreClasses     = array_values($coreClasses);
      $totalCoreClasses = count($_coreClasses);

      $formeaSubmissions = new FormeaSubmissions([
        'coreClasses' => $coreClasses,
        'formObject'  => $formObject
      ]);
      $assignedElements  = $formeaSubmissions->prepareForValidations($_elements, $data, $uniqueString);

      $totalAssignedElements = count($assignedElements);
      for ($k = 0; $k < $totalAssignedElements; $k++) {
        $formeaElement        = new FormeaElement([
          'element_id'       => $assignedElements[$k]->element_id,
          'is_preview'       => false,
          'isPreview'        => false,
          'formUniqueString' => $uniqueString,
        ]);
        if ($formObject->params->get('submission_method', 'HTTP') === 'AJAX') {
          $assignedElements[$k]->error_classes = [
            'element_container'  => $themeClass->getContainerClass(true),
            'input'              => $themeClass->getErrorClass(),
            'feedback_container' => $themeClass->getFeedbackContainerClass(true),
          ];
          $assignedElements[$k]->classes       = [
            'element_container'  => $themeClass->getContainerClass(false),
            'input'              => $themeClass->getNoErrorClass(),
            'feedback_container' => $themeClass->getFeedbackContainerClass(false),
          ];
        }

        $__key                = $assignedElements[$k]->element_id . '_' . $assignedElements[$k]->group_id . '_' . $assignedElements[$k]->setIndex;
        $assignedElements[$k] = $formeaElement->beforeValidatingSubmission($assignedElements[$k]);

        if ($assignedElements[$k]->byPassValidation) {
          $assignedElements[$k]->validation_result = null;
        } else {
          $validate = $formeaElement->validateSubmission($assignedElements[$k]->submitted_value, $assignedElements[$k]);
          if (!$validate->success) {
            throw new Exception(Text::_('COM_FORMEA_VALIDATION_PROCESS_ERROR'));
          }
          $assignedElements[$k]->set('validation_result', $validate);
          $totalValidationResults = count($validate->result);
          for ($j = 0; $j < $totalValidationResults; $j++) {
            if ($validate->result[$j]->result->result->hasError) {
              $hasGlobalError = true;
              $assignedElements[$k]->set('validation_error', true);
            }
          }
        }

        $skipValueStore = false;
        if (isset($assignedElements[$k]->skip_value_store)) {
          $skipValueStore = $assignedElements[$k]->skip_value_store;
        }

        if (!$skipValueStore) {
          $details = $formeaElement->getDetails(true, true);

          $selectedDetails = FormeaGeneralHelper::getDefaultLangValField([
            'caption',
            'placeholder',
            'description'
          ], $details, $this->langTag);

          $caption = $assignedElements[$k]->title;
          if (!empty($selectedDetails) && isset($selectedDetails['caption'])) {
            $caption = $selectedDetails['caption'];
          }

          $assignedElements[$k]->set('caption', $caption);
          $assignedElements[$k] = $formeaElement->afterValidatingSubmission($assignedElements[$k]);
          $submittedValue       = $assignedElements[$k]->submitted_value;
          if ($assignedElements[$k]->is_link > 0) {
            $submittedValue = '';
          }
          $processedElementClass[$__key] = $formeaElement;;

          //form_id,field_id,field_name, field_text, field_caption, field_value, is_link, link_path,dir_path, group_id, setIndex
          $__key                          = $assignedElements[$k]->element_id . '_' . $assignedElements[$k]->group_id . '_' . $assignedElements[$k]->setIndex;
          $submission_data_values[$__key] = [
            $this->id,
            $assignedElements[$k]->element_id,
            $assignedElements[$k]->alias,
            $assignedElements[$k]->title,
            $assignedElements[$k]->caption,
            $submittedValue,
            $assignedElements[$k]->is_link,
            json_encode($assignedElements[$k]->link_path),
            json_encode($assignedElements[$k]->dir_path),
            $assignedElements[$k]->group_id,
            $assignedElements[$k]->setIndex
          ];
        }
      }


      if ($hasGlobalError) {
        $msg = Text::_('COM_FORMEA_VALIDATION_GLOBAL_FORM_ERROR');
        if (isset($submissionMsgs->global_error_msg)) {
          if (!empty($submissionMsgs->global_error_msg)) {
            $msg = Text::_($submissionMsgs->global_error_msg);
          }
        }
        $retObject->setMsg($msg, 'danger', true);
      } else {
        //store in DB
        $submission_data_values = array_values($submission_data_values);
        $clientIP               = $_SERVER['REMOTE_ADDR'];
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
          $clientIP = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
          $clientIP = $_SERVER['HTTP_X_FORWARDED_FOR'];
        }
        //TODO:trigger before submission event plugin
        $user_id    = 0;
        $userObject = Factory::getApplication()->getIdentity();
        if (!$userObject->guest) {
          $user_id = $userObject->get('id');
        }
        $db    = $this->db;
        $query = $db->getQuery(true);
        $date  = new FsrDate();
        $query->clear();
        $submission->user_id          = $user_id;
        $submission->language         = $this->langTag;
        $submission->ip_address       = $clientIP;
        $submission->state            = 1; //determine if email sent?
        $submission->user_email_sent  = 0;
        $submission->user_email       = null;
        $submission->admin_email_sent = 0;
        $submission->admin_email      = null;
        $submission->submitted_url    = (!$unknown_submitted_url) ? $submitted_url : null;
        $submission->created_date     = $date->toSql();
        $submission->created_by       = $user_id;
        //call core onBeforeStoreSubmission
        for ($tc = 0; $tc < $totalCoreClasses; $tc++) {
          $submission = $_coreClasses[$tc]->onBeforeStoreSubmission($submission);
        }
        $db->insertObject('#__formea_submissions', $submission);
        $submission->id      = $db->insertid();
        $submissionValues    = [];
        $totalSubmissionData = count($submission_data_values);
        for ($i = 0; $i < $totalSubmissionData; $i++) {
          $totalArr = count($submission_data_values[$i]);
          $v        = [];
          for ($j = 0; $j < $totalArr; $j++) {
            $v[] = $db->quote($submission_data_values[$i][$j]);
          }
          // $v[]                = $db->quote('NULL');
          $v[]                = $db->quote($submission->id);
          $submissionValues[] = implode(',', $v);
        }
        //form_id,field_id,field_name, field_text, field_caption, field_value, is_link, link_path, dir_path, group_id, setIndex
        $submissionValueColumns = [
          'form_id',
          'field_id',
          'field_name',
          'field_text',
          'field_caption',
          'field_value',
          'is_link',
          'link_path',
          'dir_path',
          'group_id',
          'setIndex',
          'submission_id'
        ];

        $query->insert($db->quoteName('#__formea_submission_data'));
        $query->columns($submissionValueColumns);
        $query->values($submissionValues);
        $db->setQuery($query);
        $db->execute();

        $session              = $this->app->getSession();
        $submittedSessionData = $session->get('FORMEA_SUBMITTED_DATA');
        if (empty($submittedSessionData)) {
          $submittedSessionData = [];
        }

        $totalAssignedElements = count($assignedElements);
        for ($k = 0; $k < $totalAssignedElements; $k++) {
          $__key = $assignedElements[$k]->element_id . '_' . $assignedElements[$k]->group_id . '_' . $assignedElements[$k]->setIndex;
          if (isset($processedElementClass[$__key])) {
            /** @var FormeaElement $_formeaElement */
            $_formeaElement = $processedElementClass[$__key];
            if (!empty($_formeaElement) && !is_null($_formeaElement)) {
              $formObject           = $this->getObject();
              $assignedElements[$k] = $_formeaElement->afterStoringInDb($formObject, $submission->id, $assignedElements[$k]);
            }
          }
        }

        $submittedSessionData[$submission->id] = ['id' => $submission->id, 'elements' => $assignedElements, 'processed' => false];
        $session->set('FORMEA_SUBMITTED_DATA', $submittedSessionData);

        $successMsg = Text::_('COM_FORMEA_GLOBAL_SUBMISSION_MSG');
        if (isset($submissionMsgs->submission_msg)) {
          //user might not want a message, therefore we still accept an empty string
          $successMsg = Text::_($submissionMsgs->submission_msg);
        }
        $retObject->setMsg($successMsg, 'success', true);
        //Sent Notification Email
      }
      $afterSubmission = $formObject->params->get('after_submission', 'none');
      if ($afterSubmission === 'redirect') {
        $_redirectUrl = $formObject->params->get('redirect_url');
        if (!empty($_redirectUrl)) {
          $redirect = true;
          $redirectUrl = $_redirectUrl;
        }
      }
      $retObject->setSuccess(!$hasGlobalError);
      $submissionResponse = new FormeaSubmissionResponse([
        'url'            => $submitted_url,
        'hasError'       => $hasGlobalError,
        'elements'       => $assignedElements,
        'submission'     => $submission,
        'submissionMsgs' => $submissionMsgs,
        'redirect'       => $redirect,
        'redirectUrl'    => $redirectUrl,
        'afterSubmission'    => $afterSubmission,
      ]);

      for ($l = 0; $l < $totalCoreClasses; $l++) {
        $submissionResponse = $_coreClasses[$l]->onAfterSubmission($submissionResponse);
      }

      $retObject->result = $submissionResponse;
    } catch (Exception $e) {
      $retObject->setMsg($e->getMessage());
      $retObject->setSuccess(false);
    }

    return $retObject;
  }

  public function getHoneyPotElement()
  {
    $html = '';
    $html .= '<div class="formea_hp_group"><div class="formea_name_hp">';
    $html .= '<label for="formea_name">Please enter your name:</label>';
    $html .= '<input type="text" name="formea_name" id="formea_name">';
    $html .= '</div></div>';

    return $html;
  }
}
