<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_formea
 *
 * @copyright   Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @version     1.3.0
 */
defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\WebAsset\WebAssetManager;
use Joomla\Component\Formea\Site\Libraries\FormeaGridItem;
use Joomla\Component\Installer\Administrator\Model\UpdatesitesModel;
use Joomla\Database\DatabaseDriver;
use Joomla\Filesystem\Path;
use Joomla\Registry\Registry;

/**
 * Script file of Formea Component
 *
 * @since  1.0.0
 */
class Com_FormeaInstallerScript
{
  /**
   * The version to be processed/install
   * @var string
   * @since 1.0.0
   */
  private $formeaVersion = '1.3.0';

  /**
   * The version that is currently installed
   * @var string
   * @since 1.1.0
   */
  private $currentVersion = '0.0.0';

  /**
   * Minimum Joomla version to check
   *
   * @var    string
   * @since  1.0.0
   */
  private $minimumJoomlaVersion = '4.0';

  /**
   * Minimum PHP version to check
   *
   * @var    string
   * @since  1.0.0
   */
  private $minimumPHPVersion = JOOMLA_MINIMUM_PHP;


  /**
   * Method to install the extension
   *
   * @param InstallerAdapter $parent The class calling this method
   *
   * @return  boolean  True on success
   *
   * @since  1.0.0
   */
  public function install($parent): bool
  {
    $this->updateDatabaseSchema(false);
    $this->getWelcomeMessage(false);

    return true;
  }

  /**
   * Method to uninstall the extension
   *
   * @param InstallerAdapter $parent The class calling this method
   *
   * @return  boolean  True on success
   *
   * @since  1.0.0
   */
  public function uninstall($parent): bool
  {
    echo Text::_('COM_FORMEA_INSTALLERSCRIPT_UNINSTALL');

    return true;
  }

  /**
   * Method to update the extension
   *
   * @param InstallerAdapter $parent The class calling this method
   *
   * @return  boolean  True on success
   *
   * @since  1.0.0
   *
   */
  public function update($parent): bool
  {
    $this->updateDatabaseSchema(true);
    $this->getWelcomeMessage(true);

    return true;
  }

  /**
   * Function called before extension installation/update/removal procedure commences
   *
   * @param string $type The type of change (install, update or discover_install, not uninstall)
   * @param InstallerAdapter $parent The class calling this method
   *
   * @return  boolean  True on success
   *
   * @throws Exception
   * @since  1.0.0
   *
   */
  public function preflight($type, $parent): bool
  {
    if ($type !== 'uninstall') {
      // Check for the minimum PHP version before continuing
      if (!empty($this->minimumPHPVersion) && version_compare(PHP_VERSION, $this->minimumPHPVersion, '<')) {
        Log::add(
          Text::sprintf('JLIB_INSTALLER_MINIMUM_PHP', $this->minimumPHPVersion),
          Log::WARNING,
          'jerror'
        );

        return false;
      }

      // Check for the minimum Joomla version before continuing
      if (!empty($this->minimumJoomlaVersion) && version_compare(JVERSION, $this->minimumJoomlaVersion, '<')) {
        Log::add(
          Text::sprintf('JLIB_INSTALLER_MINIMUM_JOOMLA', $this->minimumJoomlaVersion),
          Log::WARNING,
          'jerror'
        );

        return false;
      }

      if ($type === 'update') {
        //Get the installed version that are going to be replaced
        $formeaManifestFile = JPATH_ROOT . '/administrator/components/com_formea/formea.xml';
        $this->currentVersion = $this->getTheInstalledVersion($formeaManifestFile);

        //v1.1.0
        $groupedElementFolder = [
          JPATH_ROOT . '/administrator/components/com_formea/src/View/GroupedElement',
          JPATH_ROOT . '/administrator/components/com_formea/src/View/GroupedElements',
          JPATH_ROOT . '/components/com_formea/src/View/GroupedElement',
          JPATH_ROOT . '/components/com_formea/src/View/ValidateForm',
        ];

        $groupedElementFiles = [
          JPATH_ROOT . '/administrator/components/com_formea/src/Controller/GroupedElementController.php',
          JPATH_ROOT . '/administrator/components/com_formea/src/Controller/GroupedElementsController.php',
          JPATH_ROOT . '/administrator/components/com_formea/src/Model/GroupedElementModel.php',
          JPATH_ROOT . '/administrator/components/com_formea/src/Model/GroupedElementsModel.php',
          JPATH_ROOT . '/administrator/components/com_formea/src/Table/GroupedElementTable.php',
          JPATH_ROOT . '/components/com_formea/src/Controller/GroupedElementController.php',
          JPATH_ROOT . '/components/com_formea/src/Controller/ValidateFormController.php',
          JPATH_ROOT . '/components/com_formea/src/Model/GroupedElementModel.php',
          JPATH_ROOT . '/components/com_formea/src/Model/ValidateFormModel.php',
        ];

        $totalGroupedElementFolders = count($groupedElementFolder);
        for ($l = 0; $l < $totalGroupedElementFolders; $l++) {
          if ($this->folderExistsCaseSensitive($groupedElementFolder[$l])) {
            try {
              \Joomla\Filesystem\Folder::delete($groupedElementFolder[$l]);
            } catch (Exception $e) {
            }
          }
        }

        $totalGroupedElementFiles = count($groupedElementFiles);
        for ($k = 0; $k < $totalGroupedElementFiles; $k++) {
          if ($this->fileExistsCaseSensitive($groupedElementFiles[$k])) {
            try {
              \Joomla\Filesystem\File::delete($groupedElementFiles[$k]);
            } catch (Exception $e) {
            }
          }
        }
      }
    }

    return true;
  }

  /**
   * Function called after extension installation/update/removal procedure commences
   *
   * @param string $type The type of change (install, update or discover_install, not uninstall)
   * @param InstallerAdapter $parent The class calling this method
   *
   * @return  boolean  True on success
   *
   * @since  1.0.0
   *
   */
  public function postflight($type, $parent)
  {
    if ($type == 'install') {
      //Upload basic elements, themes
      $this->installBasicExtensions();
    } elseif ($type == 'update') {
      $this->updateElementSchema();
      $this->updateValidationSchema();

      //clear Association Files
      if (is_file(Path::clean(JPATH_ROOT . '/administrator/components/com_formea/src/Helper/AssociationsHelper.php'))) {
        \Joomla\Filesystem\File::delete(JPATH_ROOT . '/administrator/components/com_formea/src/Helper/AssociationsHelper.php');
      }
      if (is_file(Path::clean(JPATH_ROOT . '/components/com_formea/src/Helper/AssociationHelper.php'))) {
        \Joomla\Filesystem\File::delete(JPATH_ROOT . '/components/com_formea/src/Helper/AssociationHelper.php');
      }

      //update old update server endpoint
      $checkerName = [
        'Formea Updates',
        'Formea Package Updates',
        'Formea System Plugin Updates',
        'formea_form'
      ];
      $totalCheckerName = count($checkerName);
      $container = Factory::getContainer();
      /** @var DatabaseDriver $db */
      $db = $container->get('DatabaseDriver');
      $query = $db->getQuery(true);
      $query->select('location');
      $query->from($db->quoteName('#__update_sites'));
      $query->where($db->quoteName('type') . ' = ' . $db->quote('extension'));
      for ($i = 0; $i < $totalCheckerName; $i++) {
        if ($i == 0) {
          $query->where($db->quoteName('name') . ' LIKE ' . $db->quote('%' . $checkerName[$i] . '%'));
        } else {
          $query->orWhere($db->quoteName('name') . ' LIKE ' . $db->quote('%' . $checkerName[$i] . '%'));
        }
      }
      $db->setQuery($query);
      $updateSites = $db->loadColumn();
      $triggerUpdate = false;
      $needle = "https://feseur.com/";
      $totalUpdateSites = count($updateSites);
      for ($j = 0; $j < $totalUpdateSites; $j++) {
        // Using strpos to check if $needle is present in $stringToCheck
        if (strpos($updateSites[$j], $needle) !== false) {
          $triggerUpdate = true;
          break;
        }
      }
      if ($triggerUpdate) {
        $updateSiteModel = new UpdatesitesModel();
        try {
          $updateSiteModel->rebuild();
        } catch (Exception $e) {
          echo $e->getMessage();
        }
      }
    } //elseif (update)

    return true;
  }

  public function updateDatabaseSchema($isUpdate)
  {
    $container = Factory::getContainer();
    /** @var DatabaseDriver $db */
    $db = $container->get('DatabaseDriver');
    $query = $db->getQuery(true);
    if ($isUpdate) {
      $sql = 'ALTER TABLE `#__formea_submission_data` CHANGE `link_path` `link_path` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NULL DEFAULT NULL';
      $db->setQuery($sql);
      $db->execute();

      $query->clear();
      //check for dir_path
      $submissionDataColumns = $db->getTableColumns('#__formea_submission_data');
      if (!isset($submissionDataColumns['dir_path'])) {
        $sql = 'ALTER TABLE `#__formea_submission_data` ADD `dir_path` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NULL DEFAULT NULL AFTER `link_path`;';
        $db->setQuery($sql);
        $db->execute();
      }

      //update 1.0.7
      $query->clear();
      $sql = 'DESCRIBE `#__formea_element_values`';
      $db->setQuery($sql);
      $fields = $db->loadAssocList('Field');

      if (isset($fields['defval'])) {
        if (strtoupper($fields['defval']['Type']) !== 'TEXT') {
          $sql = 'ALTER TABLE `#__formea_element_values` CHANGE `defval` `defval` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NULL DEFAULT NULL';
          $db->setQuery($sql);
          $db->execute();
        }
      }

      $query->clear();
      $formea_forms_columns = $db->getTableColumns('#__formea_forms');
      if (!isset($formea_forms_columns['allow_submission'])) {
        $sql = 'ALTER TABLE `#__formea_forms` ADD `allow_submission` TINYINT NOT NULL DEFAULT \'0\' AFTER `limit_submission`;';
        $db->setQuery($sql);
        $db->execute();
      }

      //end update 1.0.7

      //update 1.1.0
      $query->clear();
      //check for dir_path
      $submissionsColumns = $db->getTableColumns('#__formea_submissions');
      if (!isset($submissionsColumns['submitted_url'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `submitted_url` VARCHAR(255) NULL DEFAULT NULL AFTER `admin_email`;';
        $db->setQuery($sql);
        $db->execute();
      }
      if (!isset($submissionsColumns['submission_type'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `submission_type` TINYINT NOT NULL DEFAULT \'0\' AFTER `submitted_url`;';
        $db->setQuery($sql);
        $db->execute();
      }
      if (!isset($submissionsColumns['transaction_no'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `transaction_no` VARCHAR(255) NULL DEFAULT NULL AFTER `submission_type`;';
        $db->setQuery($sql);
        $db->execute();
      }
      if (!isset($submissionsColumns['payment_status'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `payment_status` TINYINT NOT NULL DEFAULT \'0\' AFTER `transaction_no`;';
        $db->setQuery($sql);
        $db->execute();
      }
      if (!isset($submissionsColumns['payment_method'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `payment_method` VARCHAR(255) NULL DEFAULT NULL AFTER `payment_status`;';
        $db->setQuery($sql);
        $db->execute();
      }

      // script for 1.1.0
      /*$changeDefVal = $this->isLaterVersion('1.1.0',$this->currentVersion);
                if($changeDefVal){
                    $sql = 'ALTER TABLE `#__formea_element_values` CHANGE `defval` `defval` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NULL DEFAULT NULL;';
                    $db->setQuery($sql);
                    $db->execute();
                }*/

      //end update 1.1.0

      //update 1.2.0
      $query->clear();
      $formEmailContentColumns = $db->getTableColumns('#__formea_formemailcontent');
      if (!isset($formEmailContentColumns['attachments'])) {
        $sql = 'ALTER TABLE `#__formea_formemailcontent` ADD `attachments` TEXT NULL DEFAULT NULL AFTER `email_content`;';
        $db->setQuery($sql);
        $db->execute();
      }
      $query->clear();

      //submissionColumns
      if (!isset($submissionsColumns['user_email_notes'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `user_email_notes` TEXT NULL DEFAULT NULL AFTER `admin_email`;';
        $db->setQuery($sql);
        $db->execute();
      }
      if (!isset($submissionsColumns['admin_email_notes'])) {
        $sql = 'ALTER TABLE `#__formea_submissions` ADD `admin_email_notes` TEXT NULL DEFAULT NULL AFTER `user_email_notes`;';
        $db->setQuery($sql);
        $db->execute();
      }
      //end update 1.2.0

    }
  }

  public function getWelcomeMessage($isUpdate)
  {
    $app = Factory::getApplication();
    $lang = $app->getLanguage();
    $lang->load('com_formea', JPATH_ADMINISTRATOR, 'en_GB', true);
    $app = Factory::getApplication();
    /** @var WebAssetManager $wa */
    $wa = $app->getDocument()->getWebAssetManager();
    $registrar = $wa->getRegistry();
    $registrar->addRegistryFile('media/com_formea/joomla.asset.json');
    $wa->registerAndUseStyle('com_formea.admin-formea' . Uri::root() . 'media/com_formea/css/admin.min.css');
    //$wa->useStyle('com_formea.admin-formea');

    $svg = file_get_contents(JPATH_ROOT . '/media/com_formea/img/logo-icon.svg');
    $background = Uri::root() . 'media/com_formea/img/intro-bg.webp';
    $version = Text::sprintf('COM_FORMEA_VERSION_S', $this->formeaVersion);
    $welcomeMsg = Text::_('COM_FORMEA_THANK_YOU_WELCOME');
    $onBoardMessage = Text::_('COM_FORMEA_ONBOARD_MESSAGE');
    $docLink = Text::_('COM_FORMEA_DOCUMENTATION_LINK');
    $supportLink = Text::_('COM_FORMEA_SUPPORT_LINK');
    $requestFeatureLink = Text::_('COM_FORMEA_FEATURE_REQUEST_LINK');
    $update = '';
    if ($isUpdate) {
      $update = ' <p><b class="text-success">' . Text::_('COM_FORMEA_UPDATED_SUCCESSFULLY') . '</b></p>';
    }

    $html = <<<HTML
 <div class="d-block card card-body p-0">
        <div class="row">
          <div class="col-4">
            <div class="d-flex flex-column h-100"
                 style="background: url('$background');background-repeat: no-repeat;background-size: cover">
              <div class="ps-3 pt-3">
                <div style="width: 60px;min-width:60px;">$svg</div>
              </div>
              <div class="mt-auto pb-1">
                <div class="small d-inline-block"
                     style="margin-left:5px;padding:2px 5px;background-color: rgba(255,255,255,.7)">
                 $version
                </div>
              </div>
            </div>
          </div>
          <div class="col-8">
            <div class="d-block px-3 pb-3 pt-4">
              <h5>$welcomeMsg</h5>
              $update
              $onBoardMessage
              <div class="d-flex gap-2 py-3">
                $docLink
                <div class="vr"></div>
                $supportLink
                <div class="vr"></div>
                $requestFeatureLink
              </div>
            </div>
          </div>
        </div>
      </div>
HTML;

    echo $html;
  }

  /**
   * Update Free Elements Schema
   * @since 1.0.5
   */
  protected function updateElementSchema()
  {
    $container = Factory::getContainer();
    /** @var DatabaseDriver $db */
    $db = $container->get('DatabaseDriver');

    $elementLists = [
      $db->quote('Button'),
      $db->quote('Checkbox'),
      $db->quote('FormattedText'),
      $db->quote('JoomlaCaptcha'),
      $db->quote('RadioButton'),
      $db->quote('Select'),
      $db->quote('Textarea'),
      $db->quote('Textbox'),
    ];

    $totalElements = count($elementLists);
    $elements = [];
    $elementPath = JPATH_ROOT . '/components/com_formea/plugins/element';
    //get all the manifest files and assign in to an array for DB update
    for ($i = 0; $i < $totalElements; $i++) {
      $elName = str_replace('\'', '', $elementLists[$i]);
      $elXml = $elementPath . '/' . $elName . '/' . $elName . '.xml';
      if (is_file(Path::clean($elXml))) {
        $xml = simplexml_load_file($elXml);
        if (isset($xml->version) && !empty($xml->version)) {
          $elements[$elName] = ['name' => $elName, 'version' => (string)$xml->version];
        }
      }
    }

    $query = $db->getQuery(true);
    $query->select(['id', 'name']);
    $query->from($db->quoteName('#__formea_eltypes'));
    $query->where($db->quoteName('name') . ' IN (' . implode(',', $elementLists) . ')');
    $db->setQuery($query);
    $results = $db->loadObjectList();

    $totalResults = count($results);
    for ($j = 0; $j < $totalResults; $j++) {
      if (isset($elements[$results[$j]->name])) {
        $upd = new stdClass();
        $upd->id = $results[$j]->id;
        $upd->version = $elements[$results[$j]->name]['version'];
        $db->updateObject('#__formea_eltypes', $upd, 'id');
      }
    }
  }

  /**
   * Update Free Validations Schema
   * @since 1.1.8
   */
  protected function updateValidationSchema()
  {
    $container = Factory::getContainer();
    /** @var DatabaseDriver $db */
    $db = $container->get('DatabaseDriver');

    $elementLists = [
      $db->quote('RequiredFieldValidation'),
      $db->quote('EmailValidation'),
    ];

    $totalElements = count($elementLists);
    $elements = [];
    $elementPath = JPATH_ROOT . '/components/com_formea/plugins/validation';
    //get all the manifest files and assign in to an array for DB update
    for ($i = 0; $i < $totalElements; $i++) {
      $elName = str_replace('\'', '', $elementLists[$i]);
      $elXml = $elementPath . '/' . $elName . '/' . $elName . '.xml';
      if (is_file(Path::clean($elXml))) {
        $xml = simplexml_load_file($elXml);
        if (isset($xml->version) && !empty($xml->version)) {
          $elements[$elName] = ['name' => $elName, 'version' => (string)$xml->version];
        }
      }
    }

    $query = $db->getQuery(true);
    $query->select(['id', 'name']);
    $query->from($db->quoteName('#__formea_validations'));
    $query->where($db->quoteName('name') . ' IN (' . implode(',', $elementLists) . ')');
    $db->setQuery($query);
    $results = $db->loadObjectList();

    $totalResults = count($results);
    for ($j = 0; $j < $totalResults; $j++) {
      if (isset($elements[$results[$j]->name])) {
        $upd = new stdClass();
        $upd->id = $results[$j]->id;
        $upd->version = $elements[$results[$j]->name]['version'];
        $db->updateObject('#__formea_validations', $upd, 'id');
      }
    }
  }

  /**
   * Method to install the required elements, validations, themes, cores
   * @return true
   *
   * @throws Exception
   * @since 1.0.0
   */
  protected function installBasicExtensions()
  {
    $app = Factory::getApplication();
    $user = $app->getIdentity();
    $date = new Date();

    $elementTypes = [
      [
        "name" => "Textbox",
        "title" => "Textbox field",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.1",
        "value_type" => 0,
        "description" => "TBE_FORMEA_DESC",
      ],
      [
        "name" => "Button",
        "title" => "Button field",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.4",
        "value_type" => -1,
        "description" => "BUTTON_FORMEA_DESC",
      ],
      [
        "name" => "Checkbox",
        "title" => "Checkbox",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.1",
        "value_type" => 2,
        "description" => "CHECKBOX_FORMEA_DESC",
      ],
      [
        "name" => "FormattedText",
        "title" => "Formatted Text",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.0",
        "value_type" => 4,
        "description" => "FORMATTEDTEXT_FORMEA_DESC",
      ],
      [
        "name" => "JoomlaCaptcha",
        "title" => "Joomla Captcha",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.1.0",
        "value_type" => -1,
        "description" => "JOOMLACAPTCHA_FORMEA_DESC",
      ],
      [
        "name" => "RadioButton",
        "title" => "Radio Button",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.1",
        "value_type" => 1,
        "description" => "RBTN_FORMEA_DESC",
      ],
      [
        "name" => "Select",
        "title" => "SDD_FORMEA_TITLE",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.0",
        "value_type" => 2,
        "description" => "SDD_FORMEA_DESC",
      ],
      [
        "name" => "Textarea",
        "title" => "Textarea field",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.0",
        "value_type" => 0,
        "description" => "TAE_FORMEA_DESC",
      ],
    ];
    $container = Factory::getContainer();
    /** @var DatabaseDriver $db */
    $db = $container->get('DatabaseDriver');
    $totalElements = count($elementTypes);
    for ($i = 0; $i < $totalElements; $i++) {
      $elementTypes[$i]['state'] = 1;
      $elementTypes[$i]['created_date'] = $date->toSql();
      $elementTypes[$i]['created_by'] = (!is_null($user->id)) ? $user->id : 0;
      $obj = (object)$elementTypes[$i];
      $db->insertObject('#__formea_eltypes', $obj);
    }

    $themes = [
      [
        "name" => "Bootstrap5",
        "title" => "Bootstrap 5",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" => "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.0",
        "description" => "BOOTSTRAP5_FORMEA_DESC",
      ],
    ];

    $totalThemes = count($themes);
    $themeId = 0;
    for ($i = 0; $i < $totalThemes; $i++) {
      $themes[$i]['state'] = 1;
      $themes[$i]['created_date'] = $date->toSql();
      $themes[$i]['created_by'] = (!is_null($user->id)) ? $user->id : 0;
      $obj = (object)$themes[$i];
      $db->insertObject('#__formea_themes', $obj);
      if ($themeId < 1) {
        $themeId = $db->insertid();
      }
    }

    // validations
    $validations = [
      [
        "name" => "EmailValidation",
        "title" => "Email Validation",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" =>
        "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.0",
        "description" => "EMAILVALIDATION_FORMEA_DESC",
      ],
      [
        "name" => "RequiredFieldValidation",
        "title" => "Required Field",
        "author" => "Feseur Sdn Bhd",
        "creation_date" => "2022-09-01",
        "copyright" =>
        "Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.",
        "license" => "GNU General Public License version 2 or later;",
        "author_email" => "hello@feseur.com",
        "author_url" => "www.feseur.com",
        "version" => "1.0.1",
        "description" => "RFV_FORMEA_DESC",
      ],
    ];

    $totalValidations = count($validations);
    for ($i = 0; $i < $totalValidations; $i++) {
      $validations[$i]['state'] = 1;
      $validations[$i]['created_date'] = $date->toSql();
      $validations[$i]['created_by'] = (!is_null($user->id)) ? $user->id : 0;
      $obj = (object)$validations[$i];
      $db->insertObject('#__formea_validations', $obj);
    }

    $this->createDemoForm($themeId);

    return true;
  }

  protected function createDemoForm($themeId)
  {
    $container = Factory::getContainer();
    /** @var \Joomla\Database\DatabaseDriver $db */
    $db = $container->get('DatabaseDriver');
    //create element
    $samplePath = JPATH_ROOT . '/administrator/components/com_formea/src/Helper';
    $elementCode = [
      "[SP_CONTACT_INTRO]" => 0,
      "[SP_YOUR_NAME]" => 0,
      "[SP_COMPANY]" => 0,
      "[SP_EMAIL_ADDRESS]" => 0,
      "[SP_PHONE_NUMBER]" => 0,
      "[SP_ENQUIRY_TYPE]" => 0,
      "[SP_SUBJECT]" => 0,
      "[SP_MESSAGE]" => 0,
      "[SP_SUBMIT_BUTTON]" => 0,
      "[SP_THEME_ID]" => $themeId,
    ];

    //get Element samples
    $elementJson = file_get_contents($samplePath . '/SampleElements.json');
    $sampleElements = json_decode($elementJson);
    $totalSampleElements = count($sampleElements);
    $date = new Date();
    /** @var CMSApplicationInterface $app */
    $app = Factory::getApplication();
    $lang = $app->getLanguage();
    $langTag = $lang->getTag();
    $langTitle = $lang->getName();
    $defaultLangTag = 'en-GB';
    $user = $app->getIdentity();

    for ($j = 0; $j < $totalSampleElements; $j++) {
      $totalElementValues = count($sampleElements[$j]->values);
      $totalValidations = count($sampleElements[$j]->validations);
      $totalDetails = count($sampleElements[$j]->details);

      if ($langTag !== $defaultLangTag) {
        //add user lang tag
        if ($totalElementValues > 0) {
          $_value = clone $sampleElements[$j]->values[0];
          $_value->language = $langTag;
          $sampleElements[$j]->values[] = $_value;
          $totalElementValues = count($sampleElements[$j]->values);
        }
        if ($totalDetails > 0) {
          $_detail = clone $sampleElements[$j]->details[0];
          $_detail->language = $langTag;
          $sampleElements[$j]->details[] = $_detail;
          $totalDetails = count($sampleElements[$j]->details);
        }
      }

      $idPlaceholderKey = $sampleElements[$j]->id; //[SP_YOUR_NAME]
      $element = [
        'title' => $sampleElements[$j]->title,
        'alias' => $sampleElements[$j]->alias,
        'type' => $sampleElements[$j]->type,
        'state' => 1,
        'params' => $sampleElements[$j]->params,
        'created_date' => $date->toSql(),
        'created_by' => (!is_null($user->id)) ? $user->id : 0,
        'access_level' => $sampleElements[$j]->access_level,
      ];
      $element = (object)$element;
      $db->insertObject('#__formea_elements', $element);
      $element->id = $db->insertid();
      $elementCode[$idPlaceholderKey] = $element->id;

      for ($i = 0; $i < $totalElementValues; $i++) {
        $elementValues = [
          'element_id' => $element->id,
          'defval' => $sampleElements[$j]->values[$i]->defval,
          'option_lists' => $sampleElements[$j]->values[$i]->option_lists,
          'language' => $sampleElements[$j]->values[$i]->language,
        ];
        $elementValues = (object)$elementValues;
        $db->insertObject('#__formea_element_values', $elementValues);
        $elementValues->id = $db->insertid();
      }
      //validations

      for ($k = 0; $k < $totalValidations; $k++) {
        $invalidMessages = $sampleElements[$j]->validations[$k]->invalid_messages;
        if ($langTag !== $defaultLangTag) {
          $_validation = $sampleElements[$j]->validations[$k];
          $_invalidMessage = json_decode($_validation->invalid_messages, true);
          $_invalidMessage[$langTag] = $_invalidMessage[$defaultLangTag];
          $_validation = json_encode($_invalidMessage);
          $invalidMessages = $_validation;
        }
        $elementValidation = [
          'element_id' => $element->id,
          'validation_type' => $sampleElements[$j]->validations[$k]->validation_type,
          'params' => $sampleElements[$j]->validations[$k]->params,
          'invalid_messages' => $invalidMessages,
        ];
        $elementValidation = (object)$elementValidation;
        $db->insertObject('#__formea_element_validations', $elementValidation);
        $elementValidation->id = $db->insertid();
      }

      //details
      for ($k = 0; $k < $totalDetails; $k++) {
        $elementDetail = [
          'element_id' => $element->id,
          'caption' => $sampleElements[$j]->details[$k]->caption,
          'placeholder' => $sampleElements[$j]->details[$k]->placeholder,
          'description' => $sampleElements[$j]->details[$k]->description,
          'language' => $sampleElements[$j]->details[$k]->language,
        ];
        $elementDetail = (object)$elementDetail;
        $db->insertObject('#__formea_element_details', $elementDetail);
        $elementDetail->id = $db->insertid();
      }

      //attribute
      if (!empty($sampleElements[$j]->attribute)) {
        $elementAttribute = [
          'element_id' => $element->id,
          'class' => $sampleElements[$j]->attribute->class,
          'size' => $sampleElements[$j]->attribute->size,
          'max_size' => $sampleElements[$j]->attribute->max_size,
          'additional_attr' => $sampleElements[$j]->attribute->additional_attr,
        ];
        $elementAttribute = (object)$elementAttribute;
        $db->insertObject('#__formea_element_attributes', $elementAttribute);
        $elementAttribute->id = $db->insertid();
      }
    }

    //formObject
    $formJson = file_get_contents($samplePath . '/SampleForm.json');
    $formRowJson = file_get_contents($samplePath . '/SampleRowsCoded.json');
    foreach ($elementCode as $keyCode => $element_id) {
      $formJson = str_replace($keyCode, $element_id, $formJson);
      $formRowJson = str_replace($keyCode, $element_id, $formRowJson);
    }
    $sampleForm = json_decode($formJson);
    $formObject = [
      'title' => $sampleForm->title,
      'alias' => $sampleForm->alias,
      'state' => $sampleForm->state,
      'limit_submission' => $sampleForm->limit_submission,
      'allow_submission' => $sampleForm->allow_submission,
      'inline_css' => $sampleForm->inline_css,
      'additional_css' => $sampleForm->additional_css,
      'inline_js' => $sampleForm->inline_js,
      'additional_js' => $sampleForm->additional_js,
      'admin_column' => json_encode($sampleForm->admin_column),
      'form_type' => $sampleForm->form_type,
      'access_level' => $sampleForm->access_level,
      'params' => $sampleForm->params,
      'created_date' => $date->toSql(),
      'created_by' => (!is_null($user->id)) ? $user->id : 0,
    ];
    $formObject = (object)$formObject;
    $db->insertObject('#__formea_forms', $formObject);
    $formObject->id = $db->insertid();
    //formEmails
    $totalFormEmails = count($sampleForm->emails);
    for ($i = 0; $i < $totalFormEmails; $i++) {
      $formEmail = [
        'form_id' => $formObject->id,
        'target_type' => $sampleForm->emails[$i]->target_type,
        'from_email' => $sampleForm->emails[$i]->from_email,
        'to_email' => $sampleForm->emails[$i]->to_email,
        'cc_email' => $sampleForm->emails[$i]->cc_email,
        'bcc_email' => $sampleForm->emails[$i]->bcc_email,
        'replyto_email' => $sampleForm->emails[$i]->replyto_email,
      ];
      $formEmail = (object)$formEmail;
      $db->insertObject('#__formea_formemails', $formEmail);
      $formEmail->id = $db->insertid();
      $totalEmailContent = count($sampleForm->emails[$i]->contents);
      if ($langTag !== $defaultLangTag) {
        if ($totalEmailContent > 0) {
          $_emailContent = $sampleForm->emails[$i]->contents[0];
          $_emailContent->language = $langTag;
          $sampleForm->emails[$i]->contents[] = $_emailContent;
          $totalEmailContent = count($sampleForm->emails[$i]->contents);
        }
      }
      for ($j = 0; $j < $totalEmailContent; $j++) {
        $emailContent = [
          'form_id' => $formObject->id,
          'email_id' => $formEmail->id,
          'from_name' => $sampleForm->emails[$i]->contents[$j]->from_name,
          'replyto_name' => $sampleForm->emails[$i]->contents[$j]->replyto_name,
          'subject' => $sampleForm->emails[$i]->contents[$j]->subject,
          'email_content' => $sampleForm->emails[$i]->contents[$j]->email_content,
          'language' => $sampleForm->emails[$i]->contents[$j]->language,
        ];
        $emailContent = (object)$emailContent;
        $db->insertObject('#__formea_formemailcontent', $emailContent);
      }
    }
    //form details
    $totalFormDetails = count($sampleForm->details);
    if ($langTag !== $defaultLangTag) {
      if ($totalFormDetails > 0) {
        $_formDetail = $sampleForm->details[0];
        $_formDetail->language = $langTag;
        $sampleForm->details[] = $_formDetail;
        $totalFormDetails = count($sampleForm->details);
      }
    }
    for ($i = 0; $i < $totalFormDetails; $i++) {
      $formDetail = [
        'form_id' => $formObject->id,
        'form_heading' => $sampleForm->details[$i]->form_heading,
        'metakey' => $sampleForm->details[$i]->metakey,
        'metadesc' => $sampleForm->details[$i]->metadesc,
        'language' => $sampleForm->details[$i]->language,
      ];
      $formDetail = (object)$formDetail;
      $db->insertObject('#__formea_formdetails', $formDetail);
    }

    //form theme
    $formTheme = [
      'form_id' => $formObject->id,
      'theme_id' => $sampleForm->theme->theme_id,
      'style_id' => $sampleForm->theme->style_id,
      'load_css' => $sampleForm->theme->load_css,
      'load_js' => $sampleForm->theme->load_js,
    ];
    $formTheme = (object)$formTheme;
    $db->insertObject('#__formea_formthemes', $formTheme);

    //submissions
    $totalSubmissions = count($sampleForm->submissions);
    if ($langTag !== $defaultLangTag) {
      if ($totalFormDetails > 0) {
        $_submission = $sampleForm->submissions[0];
        $_submission->language = $langTag;
        $sampleForm->submissions[] = $_submission;
        $totalSubmissions = count($sampleForm->submissions);
      }
    }
    for ($l = 0; $l < $totalSubmissions; $l++) {
      $submission = [
        'form_id' => $formObject->id,
        'submission_msg' => $sampleForm->submissions[$l]->submission_msg,
        'global_error_msg' => $sampleForm->submissions[$l]->global_error_msg,
        'limit_reach_msg' => $sampleForm->submissions[$l]->limit_reach_msg,
        'language' => $sampleForm->submissions[$l]->language,
      ];
      $submission = (object)$submission;
      $db->insertObject('#__formea_formsubmissions', $submission);
    }

    //Rows
    $rows = json_decode($formRowJson);
    $elementLayout = new stdClass();
    $elementLayout->title = 'Section 1';
    $elementLayout->subtitle = '';
    $elementLayout->icon = '';
    $elementLayout->rows = $rows;
    $this->_storeFormLayout((int)$formObject->id, [$elementLayout], true, $langTag, $defaultLangTag, $langTitle);

    return true;
  }

  protected function _storeFormLayout($form_id, array $layoutData, $isNew = false, $langTag = 'en-GB', $defaultLangTag = 'en-GB', $langTitle = 'English (en-GB)')
  {
    if ($form_id > 0) {
      $container = Factory::getContainer();
      $db = $container->get('DatabaseDriver');
      if (!$isNew) {
        //form id already exist, clear past details
        $query = $db->getQuery(true);
        $targetTables = [
          ['table' => 'formea_form_fieldsets', 'key' => null],
          ['table' => 'formea_form_rows', 'key' => 'row_type'],
          ['table' => 'formea_form_columns', 'key' => 'column_type'],
          ['table' => 'formea_form_elements', 'key' => 'target_type'],
          ['table' => 'formea_form_groups', 'key' => 'group_type'],
        ];
        $totalTargetTables = count($targetTables);
        for ($j = 0; $j < $totalTargetTables; $j++) {
          $conditions = array(
            $db->quoteName('form_id') . ' = ' . $form_id,
          );
          if (!empty($targetTables[$j]['key'])) {
            $conditions[] = $db->quoteName($targetTables[$j]['key']) . ' = 0';
          }
          $query->clear();
          $query->delete($db->quoteName('#__' . $targetTables[$j]['table']));
          $query->where($conditions);
          $db->setQuery($query);
          $db->execute();
        }
      }
      $totalElement = count($layoutData);
      for ($i = 0; $i < $totalElement; $i++) {
        $formPage = (object)[
          'form_id' => $form_id,
          'title' => $layoutData[$i]->title,
          'subtitle' => $layoutData[$i]->subtitle,
          'icon' => $layoutData[$i]->icon,
          'settings' => json_encode([]),
        ];
        $db->insertObject('#__formea_form_fieldsets', $formPage);
        $formPage->id = $db->insertid();
        $styleSheetParams = $this->_getStyleSheetParams();
        $structure = $this->_storeFormeaGridItem($form_id, $formPage->id, $layoutData[$i]->rows, $styleSheetParams, $langTag, $defaultLangTag, $langTitle);
        $upd = new stdClass();
        $upd->id = $formPage->id;
        $upd->layout = json_encode($structure);
        $db->updateObject('#__formea_form_fieldsets', $upd, 'id');
      }
    }

    return true;
  }

  protected function _storeFormeaGridItem($form_id, $formPageId, $items, $styleSheetParams, $langTag = 'en-GB', $defaultLangTag = 'en-GB', $langTitle = 'English (en-GB)')
  {
    $structure = [];
    $totalItems = count($items);
    $container = Factory::getContainer();
    $db = $container->get('DatabaseDriver');
    for ($i = 0; $i < $totalItems; $i++) {
      $targetId = 0;
      $item_type = $items[$i]->item_type;
      $settings = json_encode($items[$i]->settings);
      $set = [];
      if ($items[$i]->item_type == 3) {
        $table = '#__formea_form_groups';
        //its an element
        $groupObject = [
          'form_id' => $form_id,
          'page_id' => $formPageId,
          'group_id' => $items[$i]->group_id,
          'group_type' => 0,
        ];
        $groupObject = (object)$groupObject;
        $db->insertObject($table, $groupObject);
        $groupObject->id = $db->insertid();
        $items[$i]->id = $groupObject->id;
        // $styleSheet        = $this->_processStyleSheet($group_id, $settings, $items[$i], $styleSheetParams);
        $targetId = $groupObject->id;
        $groupId = $items[$i]->group_id;
        $setPush = [
          'form_group_id' => $targetId,
          'group_type' => $items[$i]->group_type,
          'rows' => []
        ];
        if ($items[$i]->group_type == 1) {
          //input group, store elements
          $totalElements = count($items[$i]->sets[0]->elements);
          for ($j = 0; $j < $totalElements; $j++) {
            if ($langTag !== $defaultLangTag) {
              $totalLabelSettings = count($items[$i]->sets[0]->elements[$j]->labelSettings);
              if ($totalLabelSettings > 0) {
                $_labelSetting = $items[$i]->sets[0]->elements[$j]->labelSettings[0];
                $_labelSetting->lang_code = $langTag;
                $_labelSetting->lang_title = $langTitle;
                $items[$i]->sets[0]->elements[$j]->labelSettings[] = $_labelSetting;
              }
            }
            //its an element
            $_elementObject = [
              'form_id' => $form_id,
              'page_id' => $formPageId,
              'group_id' => $groupId,
              'settings' => $settings,
              'element_id' => $items[$i]->sets[0]->elements[$j]->element_id,
              'title' => $items[$i]->sets[0]->elements[$j]->title,
              'elementTypeTitle' => $items[$i]->sets[0]->elements[$j]->elementTypeTitle,
              'type' => $items[$i]->sets[0]->elements[$j]->type,
              'labelSettings' => json_encode($items[$i]->sets[0]->elements[$j]->labelSettings),
            ];
            $_elementObject = (object)$_elementObject;
            $db->insertObject('#__formea_form_elements', $_elementObject);
          }
        } else {
          $rows = $this->_storeFormeaGridItem($form_id, $formPageId, $items[$i]->sets[0]->rows, $styleSheetParams, $langTag, $defaultLangTag);
          $setPush['rows'] = $rows;
        }
        $set[] = $setPush;
      } else if ($items[$i]->item_type == 0) {
        $table = '#__formea_form_rows';
        //its a row
        $rowObject = [
          'form_id' => $form_id,
          'page_id' => $formPageId,
          'group_id' => $items[$i]->group_id,
          'settings' => $settings,
          'class' => $items[$i]->class,
          'gutterX' => $items[$i]->gutterX,
          'gutterY' => $items[$i]->gutterY,
        ];
        $rowObject = (object)$rowObject;
        $db->insertObject($table, $rowObject);
        $rowObject->id = $db->insertid();
        $items[$i]->id = $rowObject->id;
        $styleSheet = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
        $targetId = $rowObject->id;
      } elseif ($items[$i]->item_type == 1) {
        $table = '#__formea_form_columns';
        //its a column
        $columnObject = [
          'form_id' => $form_id,
          'page_id' => $formPageId,
          'group_id' => $items[$i]->group_id,
          'settings' => $settings,
          'class' => $items[$i]->class,
          'column' => $items[$i]->column,
        ];
        $columnObject = (object)$columnObject;
        $db->insertObject($table, $columnObject);
        $columnObject->id = $db->insertid();
        $items[$i]->id = $columnObject->id;
        $styleSheet = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
        $targetId = $columnObject->id;
      } elseif ($items[$i]->item_type == 2) {
        $table = '#__formea_form_elements';
        //its an element
        if ($langTag !== $defaultLangTag) {
          $totalLabelSettings = count($items[$i]->content->labelSettings);
          if ($totalLabelSettings > 0) {
            $_labelSetting = $items[$i]->content->labelSettings[0];
            $_labelSetting->lang_code = $langTag;
            $_labelSetting->lang_title = $langTitle;
            $items[$i]->content->labelSettings[] = $_labelSetting;
          }
        }
        $elementObject = [
          'form_id' => $form_id,
          'page_id' => $formPageId,
          'group_id' => $items[$i]->group_id,
          'settings' => $settings,
          'element_id' => $items[$i]->content->element_id,
          'title' => $items[$i]->content->title,
          'elementTypeTitle' => $items[$i]->content->elementTypeTitle,
          'type' => $items[$i]->content->type,
          'labelSettings' => json_encode($items[$i]->content->labelSettings),
        ];
        $elementObject = (object)$elementObject;
        $db->insertObject($table, $elementObject);
        $elementObject->id = $db->insertid();
        $items[$i]->id = $elementObject->id;
        $styleSheet = $this->_processStyleSheet($form_id, $settings, $items[$i], $styleSheetParams);
        $targetId = $elementObject->id;
      }

      $structure[$i]['item_type'] = $item_type;
      $structure[$i]['item_id'] = $targetId;
      $structure[$i]['items'] = [];
      $structure[$i]['sets'] = [];
      if (!empty($set)) {
        $structure[$i]['sets'] = $set;
      }

      if (!empty($styleSheet) && $item_type !== 3) {
        $upd = new stdClass();
        $upd->id = $targetId;
        $upd->styleSheet = $styleSheet;
        $db->updateObject($table, $upd, 'id');
      }


      if (!empty($items[$i]->items)) {
        $structure[$i]['items'] = $this->_storeFormeaGridItem($form_id, $formPageId, $items[$i]->items, $styleSheetParams);
      }
    }

    return $structure;
  }

  /**
   * @param stdClass|string $settings
   * @param FormeaGridItem $item
   * @param array $styleSheetParams ;
   *
   * @return string
   *
   * @since 1.0.0
   */
  protected function _processStyleSheet($formId, $settings, $item, $styleSheetParams)
  {
    $settings = new Registry($settings);
    $elemType = 'row_' . $formId . '_' . $item->id;
    if ($item->item_type == 1) {
      $elemType = 'column_' . $formId . '_' . $item->id;
    } elseif ($item->item_type == 2) {
      $elemType = 'element_' . $formId . '_' . $item->content->element_id;
    } elseif ($item->item_type == 3) {
      $elemType = 'grouped_element_' . $formId . '_' . $item->group_id;
    }

    $elemSelector = '.fm_' . $elemType;
    $st = $elemSelector . '{';
    $hasRowStyle = false;

    foreach ($styleSheetParams as $target => $props) {
      $val = $settings->get($target);
      if (!empty($val)) {
        $hasRowStyle = true;
        foreach ($props as $param) {
          $st .= $param . ':' . $val . ';';
        }
      }
    }
    $st .= '}';
    if ($hasRowStyle) {
      return $st;
    } else {
      return '';
    }
  }

  protected function _getStyleSheetParams()
  {
    return [
      'paddingTop' => ['padding-top'],
      'paddingBottom' => ['padding-bottom'],
      'paddingLeft' => ['padding-left'],
      'paddingRight' => ['padding-right'],
      'borderTopWidth' => ['border-top-width'],
      'borderRightWidth' => ['border-right-width'],
      'borderBottomWidth' => ['border-bottom-width'],
      'borderLeftWidth' => ['border-left-width'],
      'borderStyle' => ['border-style'],
      'borderColor' => ['border-color'],
      'radiusTopLeft' => ['border-top-left-radius'],
      'radiusTopRight' => ['border-top-right-radius'],
      'radiusBottomLeft' => ['border-bottom-left-radius'],
      'radiusBottomRight' => ['border-bottom-right-radius'],
      'backgroundColor' => ['background-color'],
      'textColor' => ['color'],
    ];
  }

  /**
   * Check if folder exists case-sensitive
   *
   * @param $folderPath
   *
   * @return bool
   *
   * @since 1.1.0
   */
  protected function folderExistsCaseSensitive($folderPath)
  {
    $realPath = realpath(dirname($folderPath)); // Get the real path of the parent directory
    $folderName = basename($folderPath); // Get the folder name

    if ($realPath !== false) {
      // Get the list of directories in the parent directory
      $directories = scandir($realPath);
      if ($directories !== false) {
        // Check if the folder name exists in the list (case-sensitive)
        return in_array($folderName, $directories, true);
      }
    }

    return false;
  }

  /**
   * Check if file exists, case-sensitive
   *
   * @param $filePath
   *
   * @return bool
   *
   * @since 1.0.0
   */
  protected function fileExistsCaseSensitive($filePath)
  {
    $dirname = dirname($filePath); // Get the directory name
    $filename = basename($filePath); // Get the file name

    // Get the list of files in the directory matching the case of the filename
    $files = glob($dirname . DIRECTORY_SEPARATOR . $filename, GLOB_NOCHECK | GLOB_BRACE);

    // Perform a case-sensitive comparison of the filename with each file found
    foreach ($files as $file) {
      if (strcmp($filename, basename($file)) === 0) {
        return true; // Found a match (case-sensitive)
      }
    }

    return false; // No match found
  }

  /**
   * Compare version, return true if $current is newer thant $installed
   *
   * @param $current
   * @param $installed
   *
   * @return bool
   *
   * @since 1.1.0
   */
  protected function isLaterVersion($current, $installed)
  {
    $currentVersion = explode('.', $current);
    $installedVersion = explode('.', $installed);

    // Compare each segment of the version numbers
    for ($i = 0; $i < count($currentVersion); $i++) {
      if ($currentVersion[$i] > $installedVersion[$i]) {
        return true; // Current version is later
      } elseif ($currentVersion[$i] < $installedVersion[$i]) {
        return false; // Installed version is later
      }
    }

    // All segments are equal, so the versions are the same
    return false;
  }

  /**
   * Get the version of the installed manifestPath
   *
   * @param $manifestPath
   *
   * @return string
   *
   * @since 1.1.0
   */
  protected function getTheInstalledVersion($manifestPath)
  {
    $version = '0';
    // Get the path to the component's manifest file
    // $manifestPath = JPATH_ADMINISTRATOR . '/components/' . $extension_name . '/' . $componentName . '.xml';

    // Check if the manifest file exists
    if (file_exists($manifestPath)) {
      // Load the XML data from the manifest file
      $xml = simplexml_load_file($manifestPath);

      // Get the version from the XML data
      $version = (string)$xml->version;
    }

    return $version;
  }
}
