<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_formea
 *
 * @copyright   Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @version     1.3.0
 */

namespace Joomla\Component\Formea\Administrator\Field;

defined('_JEXEC') or die('Restricted access');

use Joomla\CMS\Factory;
use Joomla\CMS\Form\FormField;
use Joomla\CMS\Language\Text;
use Joomla\CMS\WebAsset\WebAssetManager;
use Joomla\Registry\Registry;
use SimpleXMLElement;

/**
 * name Form Field class for the Genejunction component
 */
class FormeaOptionValField extends FormField
{
	/**
	 * The name field type.
	 *
	 * @var    string
	 */
	protected $type = 'FormeaOptionVal';

	public function __construct($form = null)
	{
		$app = Factory::getApplication();
		/** @var WebAssetManager $wa */
		$wa = $app->getDocument()->getWebAssetManager();
		$wa->useScript('com_formea.field-option-val');
		parent::__construct($form);
	}

	public function setup(SimpleXMLElement $element, $value, $group = null)
	{

		// Make sure there is a valid FormField XML element.
		if ((string) $element->getName() !== 'field') {
			return false;
		}

		// Reset the input and label values.
		$this->input = null;
		$this->label = null;

		// Set the XML element object.
		$this->element = $element;

		// Set the group of the field.
		$this->group = $group;

		$attributes = [
			'multiple',
			'name',
			'id',
			'hint',
			'class',
			'description',
			'labelclass',
			'onchange',
			'onclick',
			'validate',
			'pattern',
			'validationtext',
			'default',
			'required',
			'disabled',
			'readonly',
			'autofocus',
			'hidden',
			'autocomplete',
			'spellcheck',
			'translateHint',
			'translateLabel',
			'translate_label',
			'translateDescription',
			'translate_description',
			'size',
			'showon',
		];

		$this->default = isset($element['value']) ? (string) $element['value'] : $this->default;

		$this->setValue($value);

		// Lets detect miscellaneous data attribute. For eg, data-*
		foreach ($this->element->attributes() as $key => $value) {
			if (strpos($key, 'data-') === 0) {
				// Data attribute key value pair
				$this->dataAttributes[$key] = $value;
			}
		}

		foreach ($attributes as $attributeName) {
			$this->__set($attributeName, $element[$attributeName]);
		}

		// Allow for repeatable elements
		$repeat       = (string) $element['repeat'];
		$this->repeat = ($repeat === 'true' || $repeat === 'multiple' || (!empty($this->form->repeat) && $this->form->repeat == 1));

		// Set the visibility.
		$this->hidden = ($this->hidden || strtolower((string) $this->element['type']) === 'hidden');

		$this->layout = !empty($this->element['layout']) ? (string) $this->element['layout'] : $this->layout;

		$this->parentclass = isset($this->element['parentclass']) ? (string) $this->element['parentclass'] : $this->parentclass;

		// Add required to class list if field is required.
		if ($this->required) {
			$this->class = trim($this->class . ' required');
		}

		return true;
	}


	/**
	 * Method to get the field input markup for the file field.
	 * Field attributes allow specification of a maximum file size and a string
	 * of accepted file extensions.
	 *
	 * @return  string  The field input markup.
	 *
	 * @note    The field does not include an upload mechanism.
	 * @see     \Joomla\CMS\Form\Field\MediaField
	 * @since   1.7.0
	 */
	protected function getInput()
	{

		$value = '[]';
		if (!empty($this->value)) {
			if (!is_string($this->value)) {
				$value = (new Registry($this->value))->toArray();
				$value = json_encode($value);
			} else {
				$value = $this->value;
			}
		}
		$languageString = [
			'label_value'    => Text::_($this->getAttribute('labelvalue', 'Value')),
			'label_text'     => Text::_($this->getAttribute('labeltext', 'Text')),
			'label_selected' => Text::_($this->getAttribute('labelselected', 'Selected')),
			'label_no'       => Text::_($this->getAttribute('labelno', 'JNO')),
			'label_yes'      => Text::_($this->getAttribute('labelyes', 'JYES')),
		];

		$limit = (int)$this->getAttribute('limit', 0);

		$multiple  = filter_var($this->getAttribute('multiple', 'true'), FILTER_VALIDATE_BOOLEAN);
		$_multiple = 'false';
		if ($multiple) {
			$_multiple = 'true';
		}

		$input = '<formea-optionval-field ';
		$input .= 'limit="' . $limit . '" ';
		$input .= "language-text='" . json_encode($languageString) . "' ";
		$input .= "value='" . $value . "' ";
		$input .= 'multiple="' . $_multiple . '" ';
		$input .= 'name="' . $this->name . '" ';
		$input .= '></formea-optionval-field>';

		return $input;
	}

	protected function getLabel()
	{
		return '';
	}

	/**
	 * Method to get the name used for the field input tag.
	 *
	 * @param   string  $fieldName  The field element name.
	 *
	 * @return  string  The name to be used for the field input tag.
	 *
	 * @since   1.7.0
	 */
	protected function getName($fieldName)
	{
		// To support repeated element, extensions can set this in plugin->onRenderSettings

		$name = '';

		// If there is a form control set for the attached form add it first.
		if ($this->formControl) {
			$name .= $this->formControl;
		}

		// If the field is in a group add the group control to the field name.
		if ($this->group) {
			// If we already have a name segment add the group control as another level.
			$groups = explode('.', $this->group);

			if ($name) {
				foreach ($groups as $group) {
					$name .= '[' . $group . ']';
				}
			} else {
				$name .= array_shift($groups);

				foreach ($groups as $group) {
					$name .= '[' . $group . ']';
				}
			}
		}

		// If we already have a name segment add the field name as another level.
		if ($name) {
			$name .= '[' . $fieldName . ']';
		} else {
			$name .= $fieldName;
		}


		return $name;
	}
}
