<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_formea
 *
 * @copyright   Copyright (C) 2010-2025 Feseur Sdn Bhd. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 * @version     1.3.0
 */

namespace Joomla\Component\Formea\Administrator\View\Coreextensions;

defined('_JEXEC') or die;

use Exception;
use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\Component\Formea\Site\Helper\FormeaGeneralHelper;
use Joomla\Filesystem\Path;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Helper\ContentHelper;
use Joomla\CMS\HTML\Helpers\Sidebar;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\Object\CMSObject;
use Joomla\CMS\Pagination\Pagination;
use Joomla\CMS\Toolbar\Toolbar;
use Joomla\CMS\Toolbar\ToolbarHelper;
use Joomla\Component\Content\Administrator\Extension\ContentComponent;
use Joomla\Component\Formea\Administrator\Helper\FormeaHelper;
use Joomla\Component\Formea\Administrator\Model\ValidationsModel;
use Joomla\Component\Formea\Administrator\View\FormeaView;

/**
 * View class for a list of formeas.
 *
 * @since  1.0.0
 */
class HtmlView extends FormeaView
{
  /**
   * An array of items
   *
   * @var    array
   * @since  1.0.0
   */
  public $items = [];

  /**
   * The pagination object
   *
   * @var    Pagination
   * @since  1.0.0
   */
  public $pagination;

  /**
   * The model state
   *
   * @var    CMSObject
   * @since  1.0.0
   */
  public $state;

  /**
   * Form object for search filters
   *
   * @var    Form
   * @since  1.0.0
   */
  public $filterForm;

  /**
   * The active search filters
   *
   * @var    array
   * @since  1.0.0
   */
  public $activeFilters = [];

  /**
   * The sidebar markup
   *
   * @var    string
   * @since  1.0.0
   */
  protected $sidebar;

  /**
   * @var CMSApplicationInterface
   * @since 1.0.0
   */
  protected $app;

  /**
   * Is this view an Empty State
   *
   * @var   boolean
   * @since 1.0.0
   */
  private $isEmptyState = false;

  /**
   * Method to display the view.
   *
   * @param   string  $tpl  A template file to load. [optional]
   *
   * @return  void
   *
   * @throws  Exception
   * @since   1.0.0
   */
  public function display($tpl = null): void
  {
    /** @var ValidationsModel $model */
    $model               = $this->getModel();
    $this->items         = $model->getItems();
    $this->pagination    = $model->getPagination();
    $this->filterForm    = $model->getFilterForm();
    $this->activeFilters = $model->getActiveFilters();
    $this->state         = $model->getState();
    $this->isEmptyState  = $this->get('IsEmptyState');
    $this->app           = Factory::getApplication();
    $errors              = $this->getErrors();

    if (count($errors)) {
      throw new GenericDataException(implode("\n", $errors), 500);
    }

    if (!count($this->items) && $this->isEmptyState) {
      $this->setLayout('emptystate');
    }

    // We don't need toolbar in the modal window.
    if ($this->getLayout() !== 'modal') {
      FormeaHelper::addSubmenu('formeas');
      $this->addToolbar();
      $this->sidebar = Sidebar::render();
    }

    $this->getDocument()->setTitle(Text::_('COM_FORMEA_CORES_MANAGER') . ' - ' . Text::_('COM_FORMEA'));
    $this->_loadLanguages();
    parent::display($tpl);
  }

  /**
   * Add the page title and toolbar.
   *
   * @return  void
   *
   * @throws  Exception
   * @since   1.0.0
   */
  protected function addToolbar()
  {
    $canDo = ContentHelper::getActions('com_formea', 'component');
    $user  = Factory::getApplication()->getIdentity();

    // Get the toolbar object instance
    $toolbar = Toolbar::getInstance('toolbar');

    ToolbarHelper::title(
      Text::_('COM_FORMEA_CORES_MANAGER'),
      'code-branch formea'
    );

    if (
      $canDo->get('core.create')
      || count($user->getAuthorisedCategories(
        'com_formea',
        'core.create'
      )) > 0
    ) {
      //$toolbar->addNew('formea.add');
    }


    if ($canDo->get('core.edit.state')) {
      $toolbar->popupButton('FormeaCorePlg')->icon('icon-plus')
        ->text(Text::_('COM_FORMEA_CORE_INSTALL_NEW'))
        ->selector('coreInstallModal')
        ->listCheck(false);

      $dropdown = $toolbar->dropdownButton('status-group')
        ->text('JTOOLBAR_CHANGE_STATUS')
        ->toggleSplit(false)
        ->icon('icon-ellipsis-h')
        ->buttonClass('btn btn-action')
        ->listCheck(true);

      $childBar = $dropdown->getChildToolbar();

      $childBar->publish('coreextensions.publish')->listCheck(true);

      $childBar->unpublish('coreextensions.unpublish')->listCheck(true);


      if ($user->authorise('core.admin')) {
        $childBar->checkin('coreextensions.checkin')->listCheck(true);
      }
      if ($this->state->get('filter.published') != ContentComponent::CONDITION_TRASHED) {
        $childBar->trash('coreextensions.trash')->listCheck(true);
      }
    }

    if (!$this->isEmptyState && $this->state->get('filter.published') == ContentComponent::CONDITION_TRASHED && $canDo->get('core.delete')) {
      $toolbar->delete('coreextensions.delete', 'JTOOLBAR_EMPTY_TRASH')
        ->message('JGLOBAL_CONFIRM_DELETE')
        ->listCheck(true);
    }

    if (
      $user->authorise('core.admin', 'com_formea')
      || $user->authorise(
        'core.options',
        'com_formea'
      )
    ) {
      $toolbar->preferences('com_formea');
    }

    ToolbarHelper::divider();
    ToolbarHelper::help('', false, FormeaHelper::getHelpLink('Cores'));

    HTMLHelper::_('sidebar.setAction', 'index.php?option=com_formea');
  }


  /**
   * Returns an array of fields the table can be sorted by
   *
   * @return  array  Array containing the field name to sort by as the key and display text as value
   *
   * @since   1.0.0
   */
  protected function getSortFields()
  {
    return [
      'a.ordering'     => Text::_('JGRID_HEADING_ORDERING'),
      'a.published'    => Text::_('JSTATUS'),
      'a.name'         => Text::_('JGLOBAL_TITLE'),
      'category_title' => Text::_('JCATEGORY'),
      'a.access'       => Text::_('JGRID_HEADING_ACCESS'),
      'a.language'     => Text::_('JGRID_HEADING_LANGUAGE'),
      'a.id'           => Text::_('JGRID_HEADING_ID'),
    ];
  }

  protected function _loadLanguages()
  {
    if (!empty($this->items)) {
      $totalItems = count($this->items);
      $lang       = $this->app->getLanguage();
      $langTag    = $lang->getTag();
      for ($i = 0; $i < $totalItems; $i++) {
        FormeaGeneralHelper::loadExtensionLanguage($this->items[$i]->name, 'core', $lang, $langTag);
      }
    }
  }
}
